package cn.com.duibaboot.ext.autoconfigure.datasource;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.core.utils.PropertyResolver;
import com.zaxxer.hikari.HikariDataSource;
import org.apache.commons.dbcp2.BasicDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanNotOfRequiredTypeException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.cloud.context.environment.EnvironmentChangeEvent;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.env.AbstractEnvironment;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;

import javax.annotation.Resource;
import javax.sql.DataSource;
import java.util.HashSet;
import java.util.Set;

import static cn.com.duibaboot.ext.autoconfigure.datasource.DataSourceBeanDefinitionRegistryPostProcessor.DUIBA_DATA_SOURCE_CONFIG_PREFIX;

@Configuration
@ConditionalOnClass({ DataSource.class })
@AutoConfigureBefore(DataSourceAutoConfiguration.class)
public class DuibaDataSourceAutoConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(DuibaDataSourceAutoConfiguration.class);

    @Resource
    private ApplicationContext applicationContext;

    /**
     * 设置HikariCP连接池的默认配置
     */
    @Configuration
    @ConditionalOnClass(HikariDataSource.class)
    public static class HikariDataSourcePostProcessorConfiguration{

        @Bean
        public static SpecifiedBeanPostProcessor<HikariDataSource> hikariDataSourcePostProcessor(){
            return new SpecifiedBeanPostProcessor<HikariDataSource>() {
                @Override
                public Class<HikariDataSource> getBeanType() {
                    return HikariDataSource.class;
                }

                @Override
                public Object postProcessBeforeInitialization(HikariDataSource bean, String beanName) throws BeansException {
                    return bean;
                }

                @Override
                public Object postProcessAfterInitialization(HikariDataSource bean, String beanName) throws BeansException {
                    bean.setLeakDetectionThreshold(120000);//2分钟后连接没有归还则认为可能泄露了，会打印一个错误日志。
                    return bean;
                }

                @Override
                public int getOrder() {
                    return -100;//order 要在 PerfTestRoutingDatasource包装前生效
                }
            };
        }
    }

    /**
     * 设置dbcp2连接池的默认配置
     */
    @Configuration
    @ConditionalOnClass(BasicDataSource.class)
    public static class Dbcp2DataSourcePostProcessorConfiguration{

        @Bean
        public static SpecifiedBeanPostProcessor<BasicDataSource> basicDataSourcePostProcessor(){
            return new SpecifiedBeanPostProcessor<BasicDataSource>() {
                @Override
                public Class<BasicDataSource> getBeanType() {
                    return BasicDataSource.class;
                }

                @Override
                public Object postProcessBeforeInitialization(BasicDataSource bean, String beanName) throws BeansException {
                    return bean;
                }

                @Override
                public Object postProcessAfterInitialization(BasicDataSource bean, String beanName) throws BeansException {
                    bean.setRemoveAbandonedOnBorrow(true);
                    bean.setRemoveAbandonedOnMaintenance(true);
                    bean.setLogAbandoned(true);
                    bean.setTestOnBorrow(false);
                    if(bean.getTimeBetweenEvictionRunsMillis() == -1L) {
                        bean.setTimeBetweenEvictionRunsMillis(90000);//多久testWhileIdle一次
                    }
                    if(bean.getMaxWaitMillis() == -1L){
                        bean.setMaxWaitMillis(3000);
                    }
                    bean.setMinEvictableIdleTimeMillis(600000);//池中的连接空闲多久后被回收,默认值就是30分钟。
                    bean.setValidationQuery("/* ping */ SELECT 1");
                    bean.setTestWhileIdle(true);
                    return bean;
                }

                @Override
                public int getOrder() {
                    return -100;//order 要在 PerfTestRoutingDatasource包装前生效
                }
            };
        }
    }

    /**
     * 当duiba.datasource动态被/refresh改变时，运行时修改连接池配置
     * @param e
     */
    @EventListener(EnvironmentChangeEvent.class)
    public void onEvent(EnvironmentChangeEvent e){
        String prefix = DUIBA_DATA_SOURCE_CONFIG_PREFIX + ".";
        Set<String> dataSourceNamesNeedChange = new HashSet<>();
        for(String key : e.getKeys()){
            if(key != null && key.startsWith(prefix)){
                String dataSourceName = key.substring(prefix.length());
                dataSourceName = dataSourceName.substring(0, dataSourceName.indexOf('.'));

                dataSourceNamesNeedChange.add(dataSourceName);
            }
        }

        for(String dataSourceName : dataSourceNamesNeedChange){
            String dataSourceBeanName = DataSourceBeanDefinitionRegistryPostProcessor.normalizationDataSourceName(dataSourceName);
            DataSource ds;
            try {
                ds = applicationContext.getBean(dataSourceBeanName, DataSource.class);
            }catch(NoSuchBeanDefinitionException e1){
                logger.warn("refresh datasource 时找不到名为：" + dataSourceBeanName + "的bean，不支持运行时添加数据源，如需添加数据源，请重启", e1);
                continue;
            }catch(BeanNotOfRequiredTypeException e2){
                logger.warn("refresh datasource 时名为：" + dataSourceBeanName + "的bean不是DataSource类型，而是："+e2.getActualType().getName()+"类型，不支持refresh.", e2);
                continue;
            }

            PropertyResolver resolver = new PropertyResolver((AbstractEnvironment)applicationContext.getEnvironment(),prefix + dataSourceName);

            DuibaDataSourceProperties.DataSourcePropertiesInner props = resolver.bindToConfig(DuibaDataSourceProperties.DataSourcePropertiesInner.class);
            props.bindProperties(dataSourceBeanName, ds);
        }
    }

    @Configuration
    @ConditionalOnClass({ DataSource.class, AbstractRoutingDataSource.class})
    public static class DuibaDataSourceMonitorAutoConfiguration {

        @Bean
        public DuibaDataSourceMonitor _duibaDataSourceMonitor() {
            return new DuibaDataSourceMonitor();
        }

        @Bean
        public DuibaDataSourceMonitorEndpoint duibaDataSourceMonitorEndpoint() {
            return new DuibaDataSourceMonitorEndpoint();
        }

        @Bean
        public DuibaDataSourceEndpoint duibaDataSourceEndpoint() {
            return new DuibaDataSourceEndpoint();
        }

        @Bean
        public ApplicationListener duibaDataSourceMonitor(DuibaDataSourceMonitor _duibaDataSourceMonitor) {
            return new ApplicationListener<MainContextRefreshedEvent>() {

                @Override
                public void onApplicationEvent(MainContextRefreshedEvent mainContextRefreshedEvent) {
                    ApplicationContext context = mainContextRefreshedEvent.getApplicationContext();
                    _duibaDataSourceMonitor.startMonitorThread(context);
                }
            };
        }
    }

}
