package cn.com.duibaboot.ext.autoconfigure.logger.logback.appender;

import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.UnsynchronizedAppenderBase;
import ch.qos.logback.core.spi.DeferredProcessingAware;
import cn.com.duiba.boot.perftest.InternalPerfTestContext;
import cn.com.duiba.boot.utils.DeflateUtils;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONPath;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.internal.DefaultTransaction;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.kafka.support.SendResult;
import org.springframework.util.concurrent.ListenableFuture;

import java.util.*;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;

public class KafkaAppender<E> extends UnsynchronizedAppenderBase<E>{

    public KafkaAppender(KafkaTemplate kafkaTemplate, Collection<KafkaAppenderProperties.Pattern> patterns){
        this.kafkaTemplate = kafkaTemplate;
        this.patterns = new ArrayList<>(patterns);
        for(KafkaAppenderProperties.Pattern p : patterns){
            if(p.getMatchJson() != null && !p.getMatchJson().isEmpty()){
                this.needCheckJson = true; break;
            }
        }

        this.setQueueSize(this.patterns.get(0).getQueueSize());//始终以第一个为准
        queue = new ArrayBlockingQueue<>(queueSize);
        this.setBlockWhenFull(this.patterns.get(0).getBlockWhenFull());//始终以第一个为准
    }

    private KafkaTemplate kafkaTemplate;
    private List<KafkaAppenderProperties.Pattern> patterns;
    private boolean needCheckJson = false;

    private Logger logger = LoggerFactory.getLogger(KafkaAppender.class);

    private BlockingQueue<RoutingData> queue;

    private int queueSize = 5120;

    //异步发送kafka消息线程
    private Thread flushQueueThread = new Thread(() -> {
        while(true){
            RoutingData routingData = null;
            try {
                routingData = queue.take();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            sendEvent(routingData);

            if(Thread.currentThread().isInterrupted()){
                //停止时发送队列中剩余消息，并flush
                while(true){
                    routingData = queue.poll();
                    if(routingData == null){
                        break;
                    }
                    sendEvent(routingData);
                }
                kafkaTemplate.flush();
                break;
            }
        }
    });

    protected void sendEvent(RoutingData routingData){
        if(routingData == null){
            return;
        }

        try {
            for (String topic : routingData.topics) {
                final long start = System.nanoTime();

                ListenableFuture<SendResult> future = kafkaTemplate.send(topic, routingData.data);//更新metadata等情况下会阻塞

                if (CatUtils.isCatEnabled()) {
                    future.addCallback(result -> {
                        final Transaction transaction = Cat.newTransaction("KafkaLogAppender", "KafkaLogAppender");

                        ((DefaultTransaction) transaction).setDurationStart(start);
                        transaction.setStatus(Message.SUCCESS);
                        transaction.complete();
                    }, ex -> {
                        final Transaction transaction = Cat.newTransaction("KafkaLogAppender", "KafkaLogAppender");

                        ((DefaultTransaction) transaction).setDurationStart(start);
                        transaction.setStatus(ex);
                        transaction.complete();
                    });
                }
            }
        }catch(Throwable e){
            logger.error("logback kafka发送失败", e);
        }
    }

    /**
     * 当缓冲队列满的时候，是否阻塞，默认为false,队列满会丢弃新消息；如果设为true，则当队列满的时候，会阻塞发送线程
     */
    private boolean blockWhenFull = false;

    @Override
    protected void append(E eventObject) {
        // this step avoids LBCLASSIC-139
        if (eventObject instanceof DeferredProcessingAware) {
            ((DeferredProcessingAware) eventObject).prepareForDeferredProcessing();
        }

        if(InternalPerfTestContext.isCurrentInPerfTestMode()){
            //压测模式不把日志发到kafka里
            return;
        }

        convertAndQueue(eventObject);
    }

    private void convertAndQueue(E eventObject){
        RoutingData routingData = convertToData(eventObject);
        if(routingData == null){
            return;
        }

        if(!blockWhenFull) {
            boolean added = queue.offer(routingData);
            if (!added) {
                logger.warn("try to sendMessage to kafka failed, because queue(with size:{}) is full, drop this message", queue.size());
            }
        }else{
            try {
                queue.put(routingData);
            } catch (InterruptedException e) {
                // Ignore
                Thread.currentThread().interrupt();
            }
        }
    }

    //may return null
    private RoutingData convertToData(E eventObject){
        String eventData;
        if(eventObject instanceof ILoggingEvent){
            eventData = ((ILoggingEvent) eventObject).getMessage();
        } else{
            logger.error("[NOTIFYME]will never be here");
            return null;
        }
        JSONObject json = null;
        if(needCheckJson){
            json = JSON.parseObject(eventData);
        }
        RoutingData routingData = new RoutingData();
        for(KafkaAppenderProperties.Pattern pattern : patterns) {
            if(matches(pattern, json)){
                routingData.topics.add(pattern.getTopic());
            }
        }

        if(routingData.topics.isEmpty()){//如果topics为空，说明不需要发送。
            return null;
        }

        //需要发送给kafka的消息一般都较大（超过2KB），压缩来提高效率。
        byte[] compressedData = DeflateUtils.compress(eventData);
        routingData.data = compressedData;

        return routingData;
    }

    private boolean matches(KafkaAppenderProperties.Pattern pattern, JSONObject json){
        if(!needCheckJson){
            return true;
        }
        if(pattern.getMatchJson() == null || pattern.getMatchJson().isEmpty()){
            return true;
        }

        for(Map.Entry<String, String> entry : pattern.getMatchJson().entrySet()){
            //JSONPath表达式求值
            if(!StringUtils.equals(JSONPath.eval(json, entry.getKey()).toString(), entry.getValue())){
                return false;
            }
        }

        return true;
    }

    @Override
    public void start() {
        super.start();
        flushQueueThread.setName("KafkaAppenderFlusher");
        flushQueueThread.start();
    }

    @Override
    public void stop() {
        super.stop();
        flushQueueThread.interrupt();
    }

    public boolean isBlockWhenFull() {
        return blockWhenFull;
    }

    public void setBlockWhenFull(boolean blockWhenFull) {
        this.blockWhenFull = blockWhenFull;
    }

    public int getQueueSize() {
        return queueSize;
    }

    public void setQueueSize(int queueSize) {
        this.queueSize = queueSize;
    }

    class RoutingData {
        /**
         * 该消息需要发送到哪些kafka topic
         */
        Set<String> topics = new HashSet<>();
        /**
         * 消息内容
         */
        byte[] data;
    }

}
