package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign;

import cn.com.duibaboot.ext.autoconfigure.core.Constants;
import cn.com.duibaboot.ext.autoconfigure.core.rpc.RpcContext;
import com.dianping.cat.Cat;
import feign.RequestInterceptor;
import feign.RequestTemplate;

import java.util.Collection;

/**
 * Feign请求拦截，在请求头里加入特殊标记，标识这是一个RPC请求，不应该被cat记录为URL请求，而应该被识别为RPC请求；加入cat分布式跟踪id，和当前应用名
 */
public class CustomRequestInterceptor implements RequestInterceptor {

    /**
     * 标识这是rpc请求
     */
    public static final String X_RPC = "X-Rpc";

    /**
     * 标识序列化方式（表示http请求和响应都使用指定的序列化方式）
     */
    public static final String X_SERIAL = "X-SERIAL";

    private DuibaFeignProperties duibaFeignProperties;

    public CustomRequestInterceptor(DuibaFeignProperties duibaFeignProperties){
        this.duibaFeignProperties = duibaFeignProperties;
    }

    @Override
    public void apply(RequestTemplate template) {

        appendArgs(template);

        template.header(X_RPC, "true");

        //附加cat分布式跟踪id
        String catRootId = RpcContext.getContext().getAttachment(Cat.Context.ROOT);
        String catChildId = RpcContext.getContext().getAttachment(Cat.Context.CHILD);
        String catParentId = RpcContext.getContext().getAttachment(Cat.Context.PARENT);
        String rpcClient = RpcContext.getContext().getSourceServiceId();
        String rpcServer = RpcContext.getContext().getTargetServiceId();
        if(catRootId != null){
            template.header(Cat.Context.ROOT, catRootId);
        }
        if(catChildId != null){
            template.header(Cat.Context.CHILD, catChildId);
        }
        if(catParentId != null){
            template.header(Cat.Context.PARENT, catParentId);
        }
        //rpc客户端名
        if(rpcClient != null){
            template.header(Constants.X_RPC_CLIENT, rpcClient);
        }
        //rpc服务端名
        if(rpcServer != null){
            template.header(Constants.X_RPC_SERVER, rpcServer);
        }

        template.removeHeader("Content-Type");
        template.removeHeader("Content-Encoding");
        template.header("Content-Encoding", "application/x-www-form-urlencoded");
    }

    /**
     * 解析Rpc args，按指定的序列化方式附加到template中
     * @param template
     */
    private void appendArgs(RequestTemplate template){
        Collection<String> coll = template.headers().get(X_SERIAL);

        //普通的FeignClient也会经过这里，所以这里判断一下，只有标记为需要用fst等二进制序列化方式的才序列化
        if(coll != null && coll.size() == 1
                && DuibaFeignProperties.DuibaFeignSerialization.typeOf(coll.iterator().next()) != DuibaFeignProperties.DuibaFeignSerialization.JSON){
            Object[] args = RpcContext.getContext().getInvokeArgs();

            if(duibaFeignProperties.getSerializationEnum()
                    == DuibaFeignProperties.DuibaFeignSerialization.JSON){
                //json 序列化方式会在Feign内部解析参数，这里不用管
            } else {
                byte[] bs = duibaFeignProperties.getSerializationEnum().serialize(args);
                if (bs != null) {
                    template.body(bs, null);//不能指定charset，如果指定了，则httpclient会转回string处理
                }
            }
        }
    }
}
