package cn.com.duibaboot.ext.autoconfigure.flowreplay;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop.IgnoreSubInvokesContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ParamComparator;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.StepDiffColumn;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.*;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Joiner;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.reflect.MethodSignature;

import java.lang.reflect.Method;
import java.util.*;

/**
 * 引流回归一些工具方法
 * Created by guoyanfei .
 * 2019-03-14 .
 */
@Slf4j
public final class FlowReplayUtils {

    private static final String DUIBA_FLOW_REPLAY_REPORT_ID = System.getenv("DUIBA_FLOW_REPLAY_REPORT_ID");

    private FlowReplayUtils() {
    }

    /**
     * 判断一个字符串是否是json
     * @param json
     * @return
     */
    public static boolean isJSONValid(String json) {
        if (json == null) {
            return false;
        }
        if (!json.startsWith("{") && !json.startsWith("[")) {
            return false;
        }
        try {
            JSONObject.parseObject(json);
        } catch (Exception e0) {
            try {
                JSONObject.parseArray(json);
            } catch (Exception e1) {
                return false;
            }
        }
        return true;
    }

    /**
     * 判断是否是回归专用环境
     * @return
     */
    public static boolean isReplayEnv() {
        return StringUtils.isNotBlank(DUIBA_FLOW_REPLAY_REPORT_ID);
    }

    /**
     * 判断报告id是否合法
     * @param reportId
     * @return
     */
    public static boolean isReportIdValid(Long reportId) {
        if (StringUtils.isBlank(DUIBA_FLOW_REPLAY_REPORT_ID)) {
            return false;
        }
        return DUIBA_FLOW_REPLAY_REPORT_ID.equals(String.valueOf(reportId));
    }

    /**
     * 根据方法，解出类似以下格式的字符串
     * RemoteConsumerService.getConsumer(Long)
     * @param method
     * @return
     */
    public static String parseApiNameByMethod(Method method) {
        StringBuilder sb = new StringBuilder();
        appendType(sb, method.getDeclaringClass());
        sb.append(".");
        sb.append(method.getName());
        sb.append("(");
        Class<?>[] parametersTypes = method.getParameterTypes();
        appendTypes(sb, parametersTypes);
        sb.append(")");
        return sb.toString();
    }

    /**
     * 调用的方法是否一样
     * @param spanMethod
     * @param currentMethod
     * @return
     */
    public static boolean isMethodEqual(String spanMethod, Method currentMethod) {
        return spanMethod.equals(currentMethod.getName());
    }

    /**
     * 调用的参数类型是否一样
     * @param spanParameterTypes
     * @param currentParameterTypes
     * @return
     */
    public static boolean isArgumentsTypesEqual(String[] spanParameterTypes, Class<?>[] currentParameterTypes) {
        for (int i = 0; i < currentParameterTypes.length; i++) {
            if (!currentParameterTypes[i].getName().equals(spanParameterTypes[i])) {
                return false;
            }
        }
        return true;
    }

    public static String stringArrayToString(String[] stringArray) {
        if (stringArray == null || stringArray.length == 0) {
            return StringUtils.EMPTY;
        }
        return Joiner.on(",").join(stringArray);
    }

    public static String classArrayToString(Class<?>[] classArray) {
        if (classArray == null || classArray.length == 0) {
            return StringUtils.EMPTY;
        }
        List<String> classNames = new ArrayList<>(classArray.length);
        for (Class<?> c : classArray) {
            classNames.add(c.getName());
        }
        return Joiner.on(",").join(classNames);
    }

    /**
     * 调用的参数值是否一样
     * @param spanParameterValues
     * @param currentParameterValues
     * @return
     */
    public static boolean isArgumentsEqual(Object[] spanParameterValues, Object[] currentParameterValues) {
        for (int i = 0; i < currentParameterValues.length; i++) {
            Object spanArg = spanParameterValues[i];
            Object arg = currentParameterValues[i];
            if (!FlowReplayUtils.isObjectEqual(spanArg, arg)) {
                return false;
            }
        }
        return true;
    }

    public static Set<StepDiffColumn> compareArray(String traceId, Integer spanIdx, Object[] spanParameterValues, Object[] currentParameterValues) {
        if (StringUtils.isBlank(traceId) || spanIdx == null) {
            return null;
        }
        ParamComparator.CompareResult compareResult = ParamComparator.compareArray(spanParameterValues, currentParameterValues);
        Set<ParamComparator.DiffItem> diffItems = compareResult.getDiffItems();
        Set<StepDiffColumn> result = new HashSet<>();
        for (ParamComparator.DiffItem it : diffItems) {
            result.add(StepDiffColumn.builder()
                                     .traceId(traceId)
                                     .spanIdx(spanIdx)
                                     .paramIdx(it.getParamIdx())
                                     .columnPath(it.getColumnPath())
                                     .build());
        }
        return result;
    }

    public static Set<StepDiffColumn> compareObject(String traceId, Integer spanIdx, Object spanParameterValue, Object currentParameterValue) {
        if (StringUtils.isBlank(traceId) || spanIdx == null) {
            return null;
        }
        ParamComparator.CompareResult compareResult = ParamComparator.compareObject(spanParameterValue, currentParameterValue);
        Set<ParamComparator.DiffItem> diffItems = compareResult.getDiffItems();
        Set<StepDiffColumn> result = new HashSet<>();
        for (ParamComparator.DiffItem it : diffItems) {
            result.add(StepDiffColumn.builder()
                                     .traceId(traceId)
                                     .spanIdx(spanIdx)
                                     .paramIdx(it.getParamIdx())
                                     .columnPath(it.getColumnPath())
                                     .build());
        }
        return result;
    }

    /**
     * 方法的返回值类型是否一样
     * @param spanReturnType
     * @param currentReturnType
     * @return
     */
    public static boolean isReturnTypeEqual(String spanReturnType, Class<?> currentReturnType) {
        if (spanReturnType == null && currentReturnType == null) {
            return true;
        }
        if (currentReturnType != null) {
            return currentReturnType.getName().equals(spanReturnType);
        }
        return false;
    }

    /**
     * 方法所在类全路径是不是一样
     * @param spanTypeFullPath
     * @param currentTypeFullPath
     * @return
     */
    public static boolean isTypeFullPathEqual(String spanTypeFullPath, String currentTypeFullPath) {
        return Objects.equals(spanTypeFullPath, currentTypeFullPath);
    }

    /**
     * 全局拦截白名单，白名单的不拦截
     * @param allArguments
     * @return
     */
    public static boolean isGlobalWhitelist(Object[] allArguments) {
        if (allArguments != null) {
            for (Object obj : allArguments) {
                if (obj == null) {
                    continue;
                }
                if (FlowReplayConstants.CANNOT_DESERIALIZE_CLASSES.contains(obj.getClass().getName())) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 判断两个对象是否一样
     * @param recordVal
     * @param replayVal
     * @return
     */
    public static boolean isObjectEqual(Object recordVal, Object replayVal) {   // NOSONAR
        if (recordVal == null && replayVal == null) {
            return true;
        }
        if (recordVal == null || replayVal == null) {
            return false;
        }
        // 如果都是时间戳，则不进行比对，认为这两值相等
        if (isColumnsTimestamp(recordVal, replayVal)) {
            return true;
        }
        if (replayVal instanceof String && ((String) replayVal).contains(FlowReplayConstants.DEFAULT_UUID)) {
            return true;
        }
        if (recordVal instanceof byte[] && replayVal instanceof byte[]) {
            if (((byte[]) replayVal).length == 32 && FlowReplayConstants.DEFAULT_UUID.equals(new String((byte[]) replayVal))) {
                return true;
            }
            return Objects.deepEquals(recordVal, replayVal);
        }
        if (recordVal instanceof byte[][] && replayVal instanceof byte[][]) {
            byte[][] recordValArray = (byte[][]) recordVal;
            byte[][] replayValArray = (byte[][]) replayVal;
            if (recordValArray.length != replayValArray.length) {
                return false;
            }
            for (int i = 0; i < replayValArray.length; i++) {
                byte[] v = replayValArray[i];
                if (v.length == 32 && FlowReplayConstants.DEFAULT_UUID.equals(new String(v))) {
                    continue;
                }
                if (!Objects.deepEquals(recordValArray[i], v)) {
                    return false;
                }
            }
            return true;
        }
        String recordJson = JSON.toJSONString(recordVal);
        String replayJson = JSON.toJSONString(replayVal);
        Object recordObj = JSON.parse(recordJson);
        Object replayObj = JSON.parse(replayJson);

        if (recordObj instanceof JSONArray && replayObj instanceof JSONArray) {
            return isJSONArrayEqual((JSONArray) recordObj, (JSONArray) replayObj);
        }
        if (recordObj instanceof JSONObject && replayObj instanceof JSONObject) {
            return isJSONObjectEqual((JSONObject) recordObj, (JSONObject) replayObj);
        }

        return recordObj.equals(replayObj);
    }

    /**
     * 判断录制的值和回归的值，是不是都是时间戳
     * @param recordValue
     * @param replayValue
     * @return
     */
    public static boolean isColumnsTimestamp(Object recordValue, Object replayValue) {
        long recordValueLong = 0;
        if (recordValue instanceof String) {
            try {
                recordValueLong = Long.parseLong((String) recordValue);
            } catch (Exception e) {
                return false;
            }
        } else if (recordValue instanceof Long) {
            recordValueLong = (Long) recordValue;
        } else {
            return false;
        }
        long replayValueLong = 0;
        if (replayValue instanceof String) {
            try {
                replayValueLong = Long.parseLong((String) replayValue);
            } catch (Exception e) {
                return false;
            }
        } else if (replayValue instanceof Long) {
            replayValueLong = (Long) replayValue;
        } else {
            return false;
        }
        return isTimestamp(recordValueLong) && isTimestamp(replayValueLong);
    }

    /**
     * 判断一个long型的数字是否是时间戳
     * long的值在10年前到100年后的的值，都认为是时间戳
     * @param t
     * @return
     */
    private static boolean isTimestamp(long t) {
        long now = System.currentTimeMillis();
        long oneYearMillis = 31536000000L;  // 一年的毫秒数
        long start = now - 10 * oneYearMillis;  // 10年前的现在
        long end = now + 100 * oneYearMillis;   // 100年后的现在
        return start < t && t < end;
    }

    private static boolean isJSONArrayEqual(JSONArray recordArray, JSONArray replayArray) { // NOSONAR
        if (recordArray.size() != replayArray.size()) {
            return false;
        }

        // 遍历录制返回值数组，每一项和回归返回值数组对应的项进行比对
        Iterator<Object> iterator = recordArray.iterator();
        for (int i = 0; iterator.hasNext(); i++) {
            Object recordValue = iterator.next();
            Object replayValue = replayArray.get(i);

            if (FlowReplayConstants.DEFAULT_UUID.equals(replayValue)) {
                continue;
            }
            // 如果都是时间戳，则不进行比对，认为这两值相等
            if (isColumnsTimestamp(recordValue, replayValue)) {
                continue;
            }

            String recordValueStr = JSON.toJSONString(recordValue);
            if (FlowReplayUtils.isJSONValid(recordValueStr)) { // 如果是jsonObj或者jsonArray，那么再判断内部的东西
                String replayValueStr = (replayValue != null ? JSON.toJSONString(replayValue) : null);
                if (!isObjectEqual(JSON.parse(recordValueStr), JSON.parse(replayValueStr))) {
                    return false;
                }
            } else {    // 如果不是jsonObj或者jsonArray，那么直接进行值对比，如果值不一样，添加对比不一致的结果
                if (!Objects.equals(recordValue, replayValue)) {
                    return false;
                }
            }
        }
        return true;
    }

    private static boolean isJSONObjectEqual(JSONObject recordObj, JSONObject replayObj) {  // NOSONAR
        // 遍历录制返回值对象，每一个key的value和回归返回值对象对应的value进行比对
        for (Map.Entry<String, Object> entry : recordObj.entrySet()) {
            if (FlowReplayConstants.COLUMN_NAME_WHITELIST.contains(entry.getKey())) {
                continue;
            }

            Object recordValue = entry.getValue();
            Object replayValue = replayObj.get(entry.getKey());

            if (FlowReplayConstants.DEFAULT_UUID.equals(replayValue)) {
                continue;
            }
            // 如果都是时间戳，则不进行比对，认为这两值相等
            if (isColumnsTimestamp(recordValue, replayValue)) {
                continue;
            }

            String recordValueStr = (recordValue != null ? JSON.toJSONString(recordValue) : null);
            String replayValueStr = (replayValue != null ? JSON.toJSONString(replayValue) : null);
            if (FlowReplayUtils.isJSONValid(recordValueStr)) {  // 当前遍历到的value，是个jsonObj或者jsonArray，那么需要递归判断里面的内容
                if (!isObjectEqual(JSON.parse(recordValueStr), JSON.parse(replayValueStr))) {
                    return false;
                }
            } else {    // 当前遍历到的value，不是jsonObj或者jsonArray，那么直接进行值的比对，如果值不一样，添加对比不一致的结果
                if (!Objects.equals(recordValue, replayValue)) {
                    return false;
                }
            }
        }
        return true;
    }


    /**
     * 根据方法签名，解出类似以下格式的字符串
     * RemoteConsumerService.getConsumer(Long)
     * @param methodSignature
     * @return
     */
    public static String parseApiNameByMethodSignature(MethodSignature methodSignature) {
        StringBuilder sb = new StringBuilder();
        appendType(sb, methodSignature.getDeclaringType());
        sb.append(".");
        sb.append(methodSignature.getMethod().getName());
        sb.append("(");
        Class<?>[] parametersTypes = methodSignature.getParameterTypes();
        appendTypes(sb, parametersTypes);
        sb.append(")");
        return sb.toString();
    }

    private static void appendTypes(StringBuilder sb, Class<?>[] types) {
        for (int size = types.length, i = 0; i < size; i++) {
            appendType(sb, types[i]);
            if (i < size - 1) {
                sb.append(",");
            }
        }
    }

    private static void appendType(StringBuilder sb, Class<?> type) {
        if (type.isArray()) {
            appendType(sb, type.getComponentType());
            sb.append("[]");
        } else {
            sb.append(type.getSimpleName());
        }
    }

    /**
     * 见如下代码：
     *     public String testCustomizeCallback() throws Throwable {
     *         return FlowReplayUtils.executeCustomizeCallbackSpan("testKey", () -> function());
     *     }
     * 用这种方式包裹的function()，将会作为一个引流回归录制和回归的整体。
     * 功能类似 @FlowReplayCustomizeSpan 
     * @param key 需要保证key在项目中的唯一性
     * @param callback
     * @param <T>
     * @return
     * @throws Throwable
     */
    public static <T> T executeCustomizeCallbackSpan(String key, Callback<T> callback) throws Throwable {
        // key没传，不走引流回归逻辑
        if (StringUtils.isBlank(key)) {
            return callback.invoke();
        }
        // 如果是录制
        if (FlowReplayTrace.isTraced()) {
            return recordCustomizeCallbackSpan(key, callback);
        }
        // 如果是回归
        if (FlowReplayUtils.isReplayEnv() && ReplayTraceContext.isReplaying()) {
            return repalyCustomizeCallbackSpan(key, callback);
        }
        return callback.invoke();
    }

    private static <T> T repalyCustomizeCallbackSpan(String key, Callback<T> callback) throws Throwable {
        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        // 创建回归的span，用于回归详情的构建
        CustomizeCallbackFlowReplaySpan replayDetailSpan = createReplayDetailSpan(key, span);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.CUSTOMIZE != span.getSpanType()) {
            String expert = SpanType.CUSTOMIZE.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        CustomizeCallbackFlowReplaySpan customizeCallbackSpan = (CustomizeCallbackFlowReplaySpan) span;
        if (!key.equals(customizeCallbackSpan.getKey())) {
            String expert = customizeCallbackSpan.getKey();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_800, expert, key);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        T t = (T) customizeCallbackSpan.getReturnValue();

        // 参数都相同的情况下，mock之前把mock的返回值回填到构建回归详情用的span中
        replayDetailSpan.setRet(t);
        return t;
    }

    private static CustomizeCallbackFlowReplaySpan createReplayDetailSpan(String key, FlowReplaySpan span) {
        CustomizeCallbackFlowReplaySpan replayDetailSpan = CustomizeCallbackFlowReplaySpan.createSpan(key, null);
        replayDetailSpan.setTraceId(FlowReplayTrace.getCurrentTraceId());
        if (span != null) {
            replayDetailSpan.setSpanId(span.getSpanId());
        }
        FlowReplayTrace.addSubSpan(replayDetailSpan);
        return replayDetailSpan;
    }

    private static <T> T recordCustomizeCallbackSpan(String key, Callback<T> callback) throws Throwable {
        // 本次调用已经被标记为需要忽略，不录制
        if (IgnoreSubInvokesContext.isMarked()) {
            return callback.invoke();
        }
        // 标记本次调用的子调用不需要录制
        IgnoreSubInvokesContext.keyMark(key);

        T ret = callback.invoke();

        try {
            CustomizeCallbackFlowReplaySpan span = CustomizeCallbackFlowReplaySpan.createSpan(key, ret);
            span.setTraceId(FlowReplayTrace.getCurrentTraceId());
            FlowReplayTrace.addSubSpan(span);
        } catch (Throwable t) {
            log.error("CustomizeCallback_录制异常", t);
            // 录制异常，这个用例不录了
            FlowReplayTrace.remove();
        } finally {
            IgnoreSubInvokesContext.unmark();
        }

        return ret;
    }

    /**
     * 回调接口
     *
     * @param <R>
     */
    public interface Callback<R> {

        R invoke() throws Throwable;
    }
}
