package cn.com.duibaboot.ext.autoconfigure.flowreplay.record;

import cn.com.duiba.wolf.entity.Pair;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayConstants;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.event.RecordEndEvent;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.event.RecordStartEvent;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.serializer.Hessian2Serializer;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * trace从队列中取出来，写到文件中
 * Created by guoyanfei .
 * 2019-04-26 .
 */
@Slf4j
public class RecordTraceWriter {

    private volatile Thread traceWriteThread;

    @EventListener(RecordStartEvent.class)
    public void recordStartEventListener(RecordStartEvent event) {
        traceWriteThread = new Thread(new TraceWriteThread(RecordContextHolder.getRecordContext()), "RecordTraceWriterThread");
        traceWriteThread.start();
    }

    @EventListener(RecordEndEvent.class)
    public void recordEndEventListener() {
        if (traceWriteThread != null) {
            traceWriteThread.interrupt();
        }
    }

    private static class TraceWriteThread implements Runnable {

        private RecordContext context;

        private TraceWriteThread(RecordContext context) {
            this.context = context;
        }

        @Override
        public void run() {
            try {
                writeTrace();
            } catch (Exception e) {
                log.error("writeTrace异常", e);
                RecordContextHolder.abnormalEnd("writeTrace异常:" + e.getMessage());
            }
        }

        private void writeTrace() throws IOException {  // NOSONAR
            File recordLocalFile = new File(FlowReplayConstants.LOCAL_RECORD_FILEPATH);
            File recordDetailLocalFile = new File(FlowReplayConstants.LOCAL_RECORD_DETAIL_FILEPATH);
            try (RandomAccessFile recordRaf = new RandomAccessFile(recordLocalFile, "rw");
                    RandomAccessFile recordDetailRaf = new RandomAccessFile(recordDetailLocalFile, "rw")) {

                // 文件最大大小，超过这个值停止录制
                int expectFileSize = context.getMaxFileSize() * 1024 * 1024;

                recordRaf.writeInt(FlowReplayConstants.RECORD_FILE_VERSION);  // 录制文件的版本号
                recordRaf.writeInt(0);    // 文件初始化，用例数设置为0

                recordDetailRaf.writeInt(FlowReplayConstants.RECORD_DETAIL_FILE_VERSION);  // 录制详情文件的版本号
                recordDetailRaf.writeInt(0);    // 用例数设置为0
                recordDetailRaf.writeLong(0);    // 索引文件的开始位置设置为0

                List<Pair> traceIdIndexList = new ArrayList<>();    // key: traceId, value: index
                int usecaseCount = 0;   // 用例数量
                while (!context.isRecordFinished()) {
                    // 时间到了，该结束了
                    if (context.isTimeToEnd()) {
                        RecordContextHolder.normalEnd();
                        continue;
                    }

                    try {
                        FlowReplayTrace trace = context.pollTrace();
                        if (trace == null) {
                            continue;
                        }

                        usecaseCount++;

                        byte[] traceBytes = Hessian2Serializer.serialize(trace);
                        recordRaf.writeInt(traceBytes.length);    // 写入单个trace的长度，读取的时候先读取长度，然后根据长度，放心读取trace
                        recordRaf.write(traceBytes);              // 写入trace

                        Map<String, String> spanStackFramesMap = context.getRecordDetailContext().getAndRemoveTraceSpanStackFrames(trace.getTraceId());
                        trace.convertToDetail(spanStackFramesMap);
                        byte[] traceDetailBytes = Hessian2Serializer.serialize(trace);

                        traceIdIndexList.add(Pair.from(trace.getTraceId(), recordDetailRaf.getFilePointer()));
                        recordDetailRaf.writeInt(traceDetailBytes.length);
                        recordDetailRaf.write(traceDetailBytes);

                        context.setCurrentFileSize(recordRaf.getFilePointer());
                        // 文件大小到了，该结束了
                        if (expectFileSize < context.getCurrentFileSize()) {
                            RecordContextHolder.normalEnd();
                        }
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    } catch (Throwable t) {
                        log.error("单个用例写入异常", t);
                    }
                    if (Thread.currentThread().isInterrupted()) {
                        break;
                    }
                }
                recordRaf.seek(4);
                recordRaf.writeInt(usecaseCount);

                long idxIndex = recordDetailRaf.getFilePointer();
                for (Pair p : traceIdIndexList) {
                    byte[] traceIdBytes = Hessian2Serializer.serialize(p.getKey());
                    recordDetailRaf.writeInt(traceIdBytes.length);
                    recordDetailRaf.write(traceIdBytes);

                    recordDetailRaf.writeLong((long) p.getValue());
                }
                recordDetailRaf.seek(4);
                recordDetailRaf.writeInt(usecaseCount); // 重新写入用例数
                recordDetailRaf.writeLong(idxIndex);    // 重新写入索引文件的开始位置
            }
        }
    }
}
