package cn.com.duibaboot.ext.autoconfigure.monitor.cache;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import com.google.common.cache.Cache;
import com.google.common.collect.Maps;
import io.prometheus.client.CollectorRegistry;
import io.prometheus.client.exporter.PushGateway;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.environment.EnvironmentChangeEvent;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.env.Environment;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.EnableAsync;

import java.lang.reflect.Field;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.annotation.Resource;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2019-07-17 15:55
 * @descript:
 * @version: 1.0
 */
@Slf4j
@Configuration
@EnableAsync
public class CacheMonitorAutoConfiguration implements InitializingBean {

    @Value("${spring.application.name}")
    private String currentAppName;

    @Value("${server.port}")
    private int httpServerPort;

    @Value("${prometheus.pushgateway.address:}")
    private String pushGatewayAddress;

    private static final CollectorRegistry collectorRegistry = CollectorRegistry.defaultRegistry;

    private static final CacheMetricsCollector cacheMetrics = new CacheMetricsCollector().register(collectorRegistry);

    @Resource
    private ApplicationContext appContext;

    private static final String localIp = NetUtils.getLocalIp();

    private final String prometheusJobName = "cache_monitor";

    private ScheduledExecutorService scheduledThreadPool = Executors.newSingleThreadScheduledExecutor(new NamedThreadFactory("duiba-pushgateway-cache", true));

    @Bean
    public static SpecifiedBeanPostProcessor cacheMonitorBeanPostProcessor(final ApplicationContext appContext,
            @Value("${spring.application.name}") String currentAppName) {
        return new SpecifiedBeanPostProcessor<Object>() {
            volatile String curEnv;
            @Override
            public int getOrder() {
                return 0;
            }

            @Override
            public Class<Object> getBeanType() {
                return Object.class;
            }

            @Override
            public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
                return bean;
            }

            @Override
            public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
                if(curEnv == null) {
                    curEnv = SpringEnvironmentUtils.getCurrentEnv(appContext);
                }
                try {
                    Class beanObjClass = bean.getClass();
                    if (beanObjClass.getAnnotation(CacheMonitorExclude.class) != null) {
                        return bean;
                    }

                    Package pkg = beanObjClass.getPackage();
                    if (pkg == null) {
                        return bean;
                    }

                    String packageName = pkg.getName();
                    if (packageName.startsWith("springfox.") || packageName.startsWith("org.")
                            || packageName.startsWith("io.") || packageName.startsWith("net.")
                            || packageName.startsWith("cn.com.duibaboot") || packageName.startsWith("com.netflix")) {
                        return bean;
                    }

                    Field[] fields = bean.getClass().getDeclaredFields();
                    if (fields.length == 0) {
                        return bean;
                    }

                    CacheMonitorAutoConfiguration.addCache(fields, beanName, bean, curEnv, currentAppName);
                }catch(Exception e){
                    log.warn("", e);
                }

                return bean;
            }
        };
    }

    @SuppressWarnings("squid:S3776")
    @Async
    @EventListener(MainContextRefreshedEvent.class)
    public void loadAllCacheBean() {
        this.pushRecord();
    }

    @EventListener(EnvironmentChangeEvent.class)
    public void changeRecord() {
        CacheMonitorManager.changeOpenRecord();
        if (this.isClose()) {
            scheduledThreadPool.shutdown();
            return;
        }

        if (scheduledThreadPool.isShutdown() || scheduledThreadPool.isTerminated()) {
            scheduledThreadPool = Executors.newSingleThreadScheduledExecutor(new NamedThreadFactory("duiba-pushgateway-cache", true));
        }

        this.pushRecord();
    }

    /**
     * 保存所有缓存变量
     *
     * @param fields
     * @param beanName
     * @param beanObj
     */
    private static void addCache(Field[] fields, String beanName, Object beanObj, String curEnv, String currentAppName) {

        for (Field field : fields) {

            if (field.getAnnotation(CacheMonitorExclude.class) != null) {
                continue;
            }

            boolean isGuava = Cache.class.isAssignableFrom(field.getType());
            boolean isCaffeine = com.github.benmanes.caffeine.cache.Cache.class.isAssignableFrom(field.getType());

            if (!isGuava && !isCaffeine) {
                continue;
            }

            field.setAccessible(true);
            String keyName = String.format("%s-%s", beanName, field.getName());

            try {

                Object cache = field.get(beanObj);
                CacheMonitorManager.saveCache(cache, keyName);
                cacheMetrics.addCache(keyName, cache, localIp, currentAppName, curEnv);
            } catch (IllegalAccessException e) {
                //ignore
            } finally {
                field.setAccessible(false);
            }

        }
    }

    /**
     * 推送数据到pushGateway
     */
    private void pushRecord() {

        if (this.isClose()) {
            return;
        }

        if (cacheMetrics.getSize() == 0) {
            return;
        }

        if (StringUtils.isBlank(pushGatewayAddress)) {
            return;
        }

        if (scheduledThreadPool.isShutdown()) {
            return;
        }

        final PushGateway pushGateway = new PushGateway(pushGatewayAddress);
        Map<String, String> groupingKey = Maps.newHashMap();
        groupingKey.put("ip", localIp);
        scheduledThreadPool.scheduleAtFixedRate(() -> {

            try {
                pushGateway.pushAdd(collectorRegistry, prometheusJobName, groupingKey);
            } catch (Exception e) {
                log.debug("推送指标到pushgateway异常", e);
            }

        }, 1, 1, TimeUnit.MINUTES);
    }

    private boolean isClose() {
        Environment environment = appContext.getEnvironment();
        String isMonitorCloseVal = environment.getProperty(CacheMonitorConstant.CACHE_MONITOR_CLOSE_KEY);
        return Boolean.parseBoolean(isMonitorCloseVal);
    }

    @Override
    public void afterPropertiesSet() {

        Environment environment = appContext.getEnvironment();
        CacheMonitorManager.init(environment, currentAppName, httpServerPort);
    }


    @Configuration
    public static class CacheMonitorEndpointConfiguration {
        @Bean
        public CacheMonitorMvcEndpoint cacheMonitorEndpointConfiguration() {
            return new CacheMonitorMvcEndpoint();
        }
    }
}
