package cn.com.duibaboot.ext.autoconfigure.httpclient;

import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import org.apache.http.impl.client.CloseableHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * 对httpclient加入AOP:
 * 1.加入cat统计
 * 2.对httpClient加入hystrix功能. 当某个host下的http调用错误率或超时超过hystrix阈值时，对改host的调用进行熔断
 */
public class HttpClientAopPostProcessor implements SpecifiedBeanPostProcessor<CloseableHttpClient> {

	private static final Logger logger = LoggerFactory.getLogger(HttpClientAopPostProcessor.class);

	@Resource
	private ApplicationContext applicationContext;

	@Override
	public Class<CloseableHttpClient> getBeanType() {
		return CloseableHttpClient.class;
	}

	@Override
	public Object postProcessBeforeInitialization(CloseableHttpClient bean, String beanName) throws BeansException {
		return bean;
	}

	@Override
	public Object postProcessAfterInitialization(CloseableHttpClient bean, String beanName) throws BeansException {
		List<HttpClientMethodInterceptor> interceptors = new ArrayList<>(applicationContext.getBeansOfType(HttpClientMethodInterceptor.class).values());
		if(interceptors.isEmpty()){
			logger.warn("没有找到HttpClientMethodInterceptor的实现，不会给httpClient附加aop");
			return bean;
		}

		ProxyFactory proxyFactory = new ProxyFactory();
		proxyFactory.setTarget(bean);

		Collections.sort(interceptors, (o1, o2) -> o1.getOrder() - o2.getOrder());
		for(HttpClientMethodInterceptor interceptor : interceptors) {
			proxyFactory.addAdvice(interceptor);
		}

		return proxyFactory.getProxy();
	}

	@Override
	public int getOrder() {
		return 0;
	}

}
