package cn.com.duibaboot.ext.autoconfigure.core.utils;

import cn.com.duiba.boot.cat.CatTools;
import cn.com.duibaboot.ext.autoconfigure.cat.context.CatInstance;
import cn.com.duibaboot.ext.autoconfigure.cat.context.CatConstants;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Event;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.spi.MessageTree;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CatUtils {

    private static final Logger logger = LoggerFactory.getLogger(CatUtils.class);

    private CatUtils(){}

    public static void completeTransaction(Transaction transaction){
        if(logger.isDebugEnabled()) {
            //下面这些代码在transaction.complete内部会调用，打印日志时需要提前调用，以打印出messageId，不影响cat的实际输出
            MessageTree tree = Cat.getManager().getThreadLocalMessageTree();
            String messageId = tree.getMessageId();

            if (messageId == null) {
                messageId = Cat.createMessageId();
                tree.setMessageId(messageId);
            }

            printTransactionLog(transaction, tree);
        }

        transaction.complete();
    }

    private static void printTransactionLog(Transaction t, MessageTree tree){
        if(logger.isDebugEnabled()) {
            logger.debug("mid:{},Cat_type:{},name:{},pid:{},rid:{},thread:{},created:{}", tree.getMessageId(), t.getType(), t.getName(), tree.getParentMessageId(), tree.getRootMessageId(),
                    tree.getThreadName(), t.getTimestamp());
        }
    }

    public static boolean isCatEnabled(){
        return isCatClassExists() && CatInstance.isEnable();
    }

    public static boolean isCatClassExists(){
        return CatTools.isCatClassExists();
    }


    /**
     * 对callback内部执行的代码加上cat监控
     *
     * @param callback 要监控耗时的代码段
     * @param type
     * @param name
     * @param <T>
     * @return
     */
    public static <T> T executeInCatTransaction(Callback<T> callback, String type, String name) throws Throwable{
        if(!CatUtils.isCatEnabled()){
            return callback.invoke();
        }

        Transaction transaction = Cat.newTransaction(type, name);
        try {
            T result = callback.invoke();
            transaction.setStatus(Message.SUCCESS);
            return result;
        } catch (Throwable e) {
            Cat.logError(e);
            transaction.setStatus(e);
            throw e;
        }finally{
            transaction.complete();
        }
    }

    /**
     * 回调接口
     * @param <R>
     */
    public interface Callback<R>{
        R invoke() throws Throwable;
    }

    /**
     * 记录一个异步事件
     * @param ctx
     */
    public static void logAsyncCall(Cat.Context ctx){
        if (isCatEnabled()) {
            try {
                MessageTree tree = Cat.getManager().getThreadLocalMessageTree();
                String messageId = tree.getMessageId();
                if (messageId == null) {
                    messageId = Cat.getProducer().createMessageId();
                    tree.setMessageId(messageId);
                }
                String childId = Cat.getProducer().createRpcServerId("default");
                Cat.logEvent(CatConstants.TYPE_REMOTE_CALL, CatConstants.THREAD_POOL_CALLABLE, Event.SUCCESS, childId);
                String root = tree.getRootMessageId();
                if (root == null) {
                    root = messageId;
                }
                ctx.addProperty(Cat.Context.ROOT, root);
                ctx.addProperty(Cat.Context.PARENT, messageId);
                ctx.addProperty(Cat.Context.CHILD, childId);
            } catch (Exception e) {
                //ignore
            }
        }
    }

    public static Transaction newCompletedTransaction(String type, String name, long startNanoTime){
        try {
            if (isCatEnabled()) {
                Transaction t = Cat.getProducer().newTransaction(type, name);
                t.setDurationStart(startNanoTime);
                t.setStatus(Transaction.SUCCESS);
                t.addData("Thread", Thread.currentThread().getName());
                t.complete();
                return t;
            }
        } catch (Exception e) {
            //ignore
        }
        return null;
    }
}
