package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign;

import feign.RequestInterceptor;
import feign.RequestTemplate;
import feign.template.QueryTemplate;
import java.lang.reflect.Field;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.util.ReflectionUtils;

/**
 * 一个remote接口。
 * List<String> testList(List<String> paramList);
 * 如果paramList在客户端传的是 空数组，那么boot1和boot2会有以下两种表现
 * boot1: paramList = []
 * boot2: paramList = [null]
 *
 * 上面的情况，在feign构建httpRequest的时候
 * boot1构建的请求是：http://172.16.20.156:2222/remoteFlowReplayUserService/testList
 * boot2构建的请求是：http://172.16.20.156:2222/remoteFlowReplayUserService/testList?_p0
 *
 * 该拦截器用于将boot2中feign的该表现修改成和boot1中一致
 *
 * Created by gyf .
 * 2020/10/9 .
 */
@Slf4j
public class NewFeignJSONSerializationCompatibleCustomRequestInterceptor implements RequestInterceptor {

    private static final Field REQUEST_TEMPLATE_QUERIES_FIELD;

    static {
        Field requestTemplateQueriesField = ReflectionUtils.findField(RequestTemplate.class, "queries");
        if (requestTemplateQueriesField != null) {
            requestTemplateQueriesField.setAccessible(true);
        } else {
            log.error("RequestTemplate类中未找到queries属性，NewFeignJSONSerializationCompatibleCustomRequestInterceptor不生效，请确认feign-core的版本");
        }
        REQUEST_TEMPLATE_QUERIES_FIELD = requestTemplateQueriesField;
    }

    @Override
    public void apply(RequestTemplate template) {
        if (REQUEST_TEMPLATE_QUERIES_FIELD == null) {
            return;
        }
        Map<String, Collection<String>> queries = template.queries();
        if (queries.size() == 0) {
            return;
        }
        Map<String, Collection<String>> newQueries = new LinkedHashMap<>();
        for (Map.Entry<String, Collection<String>> entry : queries.entrySet()) {
            Collection<String> value = entry.getValue();
            if (CollectionUtils.isNotEmpty(value)) {
                newQueries.put(entry.getKey(), value);
            }
        }
        try {
            Map<String, QueryTemplate> originalQueries = (Map<String, QueryTemplate>) REQUEST_TEMPLATE_QUERIES_FIELD.get(template);
            originalQueries.clear();
        } catch (IllegalAccessException e) {
            log.error("获取RequestTemplate的原始queries异常，NewFeignJSONSerializationCompatibleCustomRequestInterceptor不生效，请联系中台解决", e);
            return;
        }
        template.queries(newQueries);
    }

}
