package cn.com.duibaboot.ext.autoconfigure.initserver;

import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.PriorityOrdered;
import org.springframework.core.annotation.AnnotatedElementUtils;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

/**
 * 扫描所有的定义BeanPostProcessor bean的地方，检查定义的方法是否为static，如否则禁止启动，以防止BeanPostProcessor依赖其他bean从而导致BeanPostProcessor列表初始化一半就初始化其他bean，导致这些bean不会被全部BeanPostProcessor处理的问题（从而可能不会被aop）。
 */
public class PriorityOrderedBeanPostProcessorChecker implements BeanDefinitionRegistryPostProcessor, PriorityOrdered {

    //顺序在 ConfigurationClassPostProcessor 之前（因为ConfigurationClassPostProcessor会把所有@Configuration注解的类进行aop，不方便获取原始信息）
    @Override
    public int getOrder() {
        return 0;
    }

    @Override
    public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry registry) throws BeansException {
        //do nothing
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        for (String beanName : beanFactory.getBeanDefinitionNames()) {
            scanBean(beanName, beanFactory);
        }
    }

    private void scanBean(String beanName, ConfigurableListableBeanFactory beanFactory){ //NOSONAR
        AbstractBeanDefinition beanDef = (AbstractBeanDefinition)beanFactory.getBeanDefinition(beanName);
        Class clazz;
        try {
            clazz = beanDef.resolveBeanClass(beanFactory.getBeanClassLoader());
        }catch(ClassNotFoundException e){
            //没有定义BeanClass
            return;
        }
        if(clazz == null){
            return;
        }
        if(AnnotatedElementUtils.hasAnnotation(clazz, Configuration.class)){
            Method[] methods = clazz.getDeclaredMethods();

            for(Method method : methods){
                if(method.isAnnotationPresent(Bean.class)
                        && (BeanPostProcessor.class.isAssignableFrom(method.getReturnType()) || SpecifiedBeanPostProcessor.class.isAssignableFrom(method.getReturnType()) || BeanFactoryPostProcessor.class.isAssignableFrom(method.getReturnType()))
                        && !Modifier.isStatic(method.getModifiers())){
                    if(!clazz.getName().startsWith("org.") && !clazz.getName().startsWith("io.")) {//对于spring自己定义的BeanPostProcessor不处理
                        throw new IllegalStateException("the method [" + clazz.getName() + "." + method.getName() + "] generated bean of type [BeanPostProcessor or SpecifiedBeanPostProcessor or BeanFactoryPostProcessor] must be static.");
                    }
                }
            }
        }
    }

}
