package cn.com.duibaboot.ext.autoconfigure.cat;

import brave.ErrorParser;
import brave.Span;
import brave.Tracer;
import org.apache.ibatis.cache.CacheKey;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.mapping.ParameterMapping;
import org.apache.ibatis.mapping.SqlCommandType;
import org.apache.ibatis.plugin.*;
import org.apache.ibatis.reflection.MetaObject;
import org.apache.ibatis.session.Configuration;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;
import org.apache.ibatis.type.TypeHandlerRegistry;
import org.springframework.context.ApplicationContext;

import javax.annotation.Resource;
import java.text.DateFormat;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Properties;

/**
 * 对MyBatis进行拦截，添加Sleuth监控
 * 
 * @author huangwq
 */
@Intercepts({
    @Signature(method = "query", type = Executor.class, args = { MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class }),
    @Signature(method = "query", type = Executor.class, args = { MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class, CacheKey.class, BoundSql.class}),
    @Signature(method = "update", type = Executor.class, args = { MappedStatement.class, Object.class }) 
})
public class SleuthMybatisPlugin implements Interceptor{

    @Resource
    private ApplicationContext applicationContext;

    //延迟加载
    private volatile Tracer tracer;
    private volatile ErrorParser errorParser;

    private Tracer getTracer(){
        if(tracer == null) {
            tracer = applicationContext.getBean(Tracer.class);
        }
        return tracer;
    }

    private ErrorParser getErrorParser(){
        if(errorParser == null) {
            errorParser = applicationContext.getBean(ErrorParser.class);
        }
        return errorParser;
    }

	@Override
	public Object intercept(Invocation invocation) throws Throwable {
        Span curSpan = getTracer().currentSpan();
        if(curSpan == null){
            return invocation.proceed();
        }

        MappedStatement mappedStatement = (MappedStatement) invocation.getArgs()[0];
        //得到 类名-方法
        String[] strArr = mappedStatement.getId().split("\\.");
        String classAndMethod = strArr[strArr.length-2] + "." + strArr[strArr.length-1];

        Span span = getTracer().nextSpan().name("sql:/"+classAndMethod).kind(Span.Kind.CLIENT)
                .remoteServiceName("MySQL")
                .start();
        Object result;
        try(Tracer.SpanInScope scope = getTracer().withSpanInScope(span)) {
            if(!span.isNoop()) {
                //获取SQL类型
                SqlCommandType sqlCommandType = mappedStatement.getSqlCommandType();
                String method = sqlCommandType.name().toLowerCase();

                span.tag("sql.method", method);//INSERT/UPDATE/...
                span.tag("sql.class_method", classAndMethod);
                span.tag("lc", "mybatis");//本地组件名
                span.tag("thread", Thread.currentThread().getName());
            }
//            span.logEvent(Span.CLIENT_SEND);

            result = invocation.proceed();
        } catch(Exception e){
            getErrorParser().error(e, span);
            throw e;
        } finally {
//            span.logEvent(Span.CLIENT_RECV);
            span.finish();
        }

        return result;
	}
	
	public String showSql(Configuration configuration, BoundSql boundSql) {
        Object parameterObject = boundSql.getParameterObject();
        List<ParameterMapping> parameterMappings = boundSql.getParameterMappings();
        String sql = boundSql.getSql().replaceAll("[\\s]+", " ");
        if (!parameterMappings.isEmpty() && parameterObject != null) {
            TypeHandlerRegistry typeHandlerRegistry = configuration.getTypeHandlerRegistry();
            if (typeHandlerRegistry.hasTypeHandler(parameterObject.getClass())) {
                sql = sql.replaceFirst("\\?", getParameterValue(parameterObject));
 
            } else {
                MetaObject metaObject = configuration.newMetaObject(parameterObject);
                for (ParameterMapping parameterMapping : parameterMappings) {
                    String propertyName = parameterMapping.getProperty();
                    if (metaObject.hasGetter(propertyName)) {
                        Object obj = metaObject.getValue(propertyName);
                        sql = sql.replaceFirst("\\?", getParameterValue(obj));
                    } else if (boundSql.hasAdditionalParameter(propertyName)) {
                        Object obj = boundSql.getAdditionalParameter(propertyName);
                        sql = sql.replaceFirst("\\?", getParameterValue(obj));
                    }
                }
            }
        }
        return sql;
    }
	
	private String getParameterValue(Object obj) {
        String value = null;
        if (obj instanceof String) {
            value = "'" + obj.toString() + "'";
        } else if (obj instanceof Date) {
            DateFormat formatter = DateFormat.getDateTimeInstance(DateFormat.DEFAULT, DateFormat.DEFAULT, Locale.CHINA);
            value = "'" + formatter.format(new Date()) + "'";
        } else {
            if (obj != null) {
                value = obj.toString();
            } else {
                value = "";
            }
        }
        return value;
    }

	@Override
	public Object plugin(Object target) {
		return Plugin.wrap(target, this);
	}

	@Override
	public void setProperties(Properties arg0) {
	    //do nothing
	}

}
