package cn.com.duibaboot.ext.autoconfigure.dubbo.group;

import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupContext;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.common.URL;
import org.apache.dubbo.rpc.Invocation;
import org.apache.dubbo.rpc.Invoker;
import org.apache.dubbo.rpc.RpcException;
import org.apache.dubbo.rpc.cluster.router.AbstractRouter;

import java.util.ArrayList;
import java.util.List;

/**
 * 用于服务分组调用的 dubbo路由器
 * 优先调用有相同【服务分组key（多场景）】的服务，如果找不到，那么调用【没有分组key（主场景）】的服务，如果还是没有，那么随便找一台调用
 * Created by gyf .
 * 2020/11/12 .
 */
public class ServiceGroupDubboRouter extends AbstractRouter {

    public ServiceGroupDubboRouter(URL url) {
        this.setUrl(url);
    }

    @Override
    public <T> List<Invoker<T>> route(List<Invoker<T>> invokers, URL url, Invocation invocation) throws RpcException {
        if (CollectionUtils.isEmpty(invokers)) {
            return invokers;
        }

        String currentServiceGroupKey = ServiceGroupContext.getCurrentGroupKey();

        List<Invoker<T>> havePriorityList = new ArrayList<>(invokers.size());  // 优先转发目标服务列表
        List<Invoker<T>> noGroupKeyInvokerList = new ArrayList<>(invokers.size());  // 没有分组标识的服务
        for (Invoker<T> invoker : invokers) {
            String invokerGroupKey = invoker.getUrl().getParameter(ServiceGroupDubboRegistryFactoryWrapper.SERVICE_GROUP_KEY);

            if (this.isPriorityGroup(currentServiceGroupKey, invokerGroupKey)) {
                // priorityGroupServer：匹配到key的所有服务列表
                havePriorityList.add(invoker);
            }

            if (this.isNoGroup(invokerGroupKey)) {
                // noGroupServer：所有serverGroupKey为空的server
                noGroupKeyInvokerList.add(invoker);
            }
        }

        // 流量优先转发给具有相同分组标识的
        if (!havePriorityList.isEmpty()) {
            return havePriorityList;
        }

        // 没有优先服务分组，且无分组服务不为空，就转发给无服务分组
        if (!noGroupKeyInvokerList.isEmpty()) {
            return noGroupKeyInvokerList;
        }

        // 真有问题的话，全部返回
        return invokers;
    }

    /**
     * 是否优先服务
     *
     * @param requestGroupKey
     * @param serverGroupKey
     * @return
     */
    private boolean isPriorityGroup(String requestGroupKey, String serverGroupKey) {
        if (StringUtils.isBlank(requestGroupKey) || StringUtils.isBlank(serverGroupKey)) {
            return false;
        }
        // 只要key相等，就在优先投放列表
        return StringUtils.equals(requestGroupKey, serverGroupKey);
    }

    /**
     * 是否无分组服务
     *
     * @param serverGroupKey
     * @return
     */
    private boolean isNoGroup(String serverGroupKey) {
        return StringUtils.isBlank(serverGroupKey);
    }

}
