/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cloud.stream.binder.rocketmq.integration;

import cn.com.duibaboot.ext.stream.binder.BinderMessageHandler;
import cn.com.duibaboot.ext.stream.support.TopicOperater;
import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.client.producer.SendCallback;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.client.producer.SendStatus;
import org.apache.rocketmq.common.message.MessageConst;
import org.apache.rocketmq.spring.core.RocketMQTemplate;
import org.apache.rocketmq.spring.support.RocketMQHeaders;
import org.springframework.cloud.stream.binder.rocketmq.metrics.Instrumentation;
import org.springframework.cloud.stream.binder.rocketmq.metrics.InstrumentationManager;
import org.springframework.integration.handler.AbstractMessageHandler;
import org.springframework.integration.support.MessageBuilder;
import org.springframework.messaging.MessagingException;
import org.springframework.util.StringUtils;

import java.util.Optional;

/**
 * @author <a href="mailto:fangjian0423@gmail.com">Jim</a>
 */
@Slf4j
public class RocketMQMessageHandler extends AbstractMessageHandler implements BinderMessageHandler {

    private final RocketMQTemplate rocketMQTemplate;

    private final String groupName;

    private final InstrumentationManager instrumentationManager;

    private boolean sync = false;

    public RocketMQMessageHandler(RocketMQTemplate rocketMQTemplate,
                                  String groupName,
                                  InstrumentationManager instrumentationManager) {
        this.rocketMQTemplate = rocketMQTemplate;
        this.groupName = groupName;
        this.instrumentationManager = instrumentationManager;
    }

    @Override
    public void start() {
        instrumentationManager.addHealthInstrumentation(new Instrumentation(groupName));
        try {
            rocketMQTemplate.afterPropertiesSet();
            instrumentationManager.getHealthInstrumentation(groupName).markStartedSuccessfully();
        }
        catch (Exception e) {
            instrumentationManager.getHealthInstrumentation(groupName).markStartFailed(e);
            log.error("RocketMQTemplate startup failed, Caused by " + e.getMessage());
            throw new MessagingException(MessageBuilder.withPayload(
                    "RocketMQTemplate startup failed, Caused by " + e.getMessage())
                    .build(), e);
        }
    }

    @Override
    public void stop() {
        rocketMQTemplate.destroy();
    }

    @Override
    protected void handleMessageInternal(org.springframework.messaging.Message<?> message){
        try {
            final StringBuilder topicWithTags = new StringBuilder(TopicOperater.findTopic(message));
            String tags = Optional
                    .ofNullable(message.getHeaders().get(RocketMQHeaders.TAGS)).orElse("")
                    .toString();
            if (!StringUtils.isEmpty(tags)) {
                topicWithTags.append(":").append(tags);
            }
            SendResult sendRes = null;
            int delayLevel = 0;
            try {
                Object delayLevelObj = message.getHeaders()
                        .getOrDefault(MessageConst.PROPERTY_DELAY_TIME_LEVEL, 0);
                if (delayLevelObj instanceof Number) {
                    delayLevel = ((Number) delayLevelObj).intValue();
                }
                else if (delayLevelObj instanceof String) {
                    delayLevel = Integer.parseInt((String) delayLevelObj);
                }
            }
            catch (Exception e) {
                // ignore
            }

            if (sync) {
                sendRes = rocketMQTemplate.syncSend(topicWithTags.toString(), message,
                        rocketMQTemplate.getProducer().getSendMsgTimeout(),
                        delayLevel);
                log.debug("sync send to topic " + topicWithTags + " " + sendRes);
            }
            else {
                rocketMQTemplate.asyncSend(topicWithTags.toString(), message,
                        new SendCallback() {
                            @Override
                            public void onSuccess(SendResult sendResult) {
                                log.debug("async send to topic " + topicWithTags + " "
                                        + sendResult);
                            }

                            @Override
                            public void onException(Throwable e) {
                                log.error(
                                        "RocketMQ Message hasn't been sent. Caused by "
                                                + e.getMessage());
                            }
                        });
            }
            if (sendRes != null && !sendRes.getSendStatus().equals(SendStatus.SEND_OK)) {
                throw new MessagingException(message,
                        new MQClientException("message hasn't been sent", null));
            }
        }
        catch (Exception e) {
            log.error("RocketMQ Message hasn't been sent. Caused by " + e.getMessage());
            throw new MessagingException(message, e);
        }
    }


    public void setSync(boolean sync) {
        this.sync = sync;
    }
}