package cn.com.duiba.cloud.biz.tool.aop.validation;

import cn.com.duiba.cloud.biz.tool.exception.BizException;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.ValidatorFactory;
import javax.validation.executable.ExecutableValidator;
import java.lang.reflect.Method;
import java.util.Set;

/**
 * remote接口参数校验的切面
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.5
 * @date 2021/12/13 5:54 下午
 * @since 0.0.5
 **/
@Aspect
public class ValidationAspect {
    private static final Validator BEAN_VALIDATOR;
    private static final ExecutableValidator METHOD_VALIDATOR;

    static {
        ValidatorFactory factory = Validation.buildDefaultValidatorFactory();
        BEAN_VALIDATOR = factory.getValidator();
        METHOD_VALIDATOR = factory.getValidator().forExecutables();
    }

    @Pointcut("@within(cn.com.duiba.cloud.biz.tool.aop.validation.Validation) || @annotation(cn.com.duiba.cloud.biz.tool.aop.validation.Validation)")
    public void validate() {

    }

    @Before("validate()")
    public void before(JoinPoint point) throws BizException {
        // 获得切入的目标对象
        Object target = point.getThis();
        // 获得切入方法参数
        Object[] args = point.getArgs();
        // 获得切入的方法
        Method method = ((MethodSignature) point.getSignature()).getMethod();
        // 校验以基础数据类型作为参数的
        Set<ConstraintViolation<Object>> validResult = validMethodParams(target, method, args);
        for (ConstraintViolation<Object> objectConstraintViolation : validResult) {
            throw new BizException(objectConstraintViolation.getMessage());
        }
        // 校验以java bean对象作为参数的
        for (Object bean : args) {
            if (null != bean) {
                validResult = validBeanParams(bean);
                for (ConstraintViolation<Object> objectConstraintViolation : validResult) {
                    throw new BizException(objectConstraintViolation.getMessage());
                }
            }
        }
    }

    private <T> Set<ConstraintViolation<T>> validMethodParams(T obj, Method method, Object[] params) {
        return METHOD_VALIDATOR.validateParameters(obj, method, params);
    }

    private <T> Set<ConstraintViolation<T>> validBeanParams(T bean) {
        return BEAN_VALIDATOR.validate(bean);
    }
}