package cn.com.duiba.cloud.biz.tool.config.datasource;

import cn.hutool.core.thread.ThreadFactoryBuilder;
import com.baomidou.dynamic.datasource.DynamicRoutingDataSource;
import com.baomidou.dynamic.datasource.ds.ItemDataSource;
import com.google.common.collect.Maps;
import com.zaxxer.hikari.HikariDataSource;
import com.zaxxer.hikari.HikariPoolMXBean;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.shardingsphere.driver.jdbc.core.datasource.ShardingSphereDataSource;
import org.apache.shardingsphere.infra.metadata.ShardingSphereMetaData;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.ApplicationContext;

import javax.sql.DataSource;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 兑吧数据源监控<br>
 * 只针对两种包装的数据源：ShardingSphereDataSource、DynamicRoutingDataSource<br>
 * 其实元数据源的类型都属于HikariDataSource，如果是别的数据源不会做监控。
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @date 2022-04-25 20:34
 * @since 0.1.0
 **/
@Slf4j
public class DuibaDataSourceMonitor implements DisposableBean {
    private boolean started = false;

    private Map<String, DataSource> dataSourceMap;

    private ExecutorService singleThreadPool;

    private static final int THRESHOLD_VALUE = 3;

    private static final String THREAD_NAME_PREFIX = "duiba-datasource-monitor-thread-";

    private static final String FORMAT_ONE = "datasource [%s]'s connectionPool is too full,max:%d,busy:%d,idle:%d";

    /**
     * HikariDataSource.getHikariPoolMXBean方法是否存在，如果不存在，代表hikari版本较低,无法监控连接池使用情况
     */
    private static final boolean HIKARI_MONITOR_EXISTS;
    private static final boolean SHARDING_CLASS_EXISTS;
    private static final boolean DYNAMIC_CLASS_EXISTS;

    static {
        boolean hikariMonitorExists = false;
        boolean shardingClassExists = false;
        boolean dynamicClassExists = false;
        try {
            HikariDataSource.class.getDeclaredMethod("getHikariPoolMXBean");
            hikariMonitorExists = true;
        } catch (NoSuchMethodException e) {
            log.warn("请把hikari升级到最新版，否则无法监控连接池使用数据");
        }

        try {
            Class.forName("com.baomidou.dynamic.datasource.DynamicRoutingDataSource");
            dynamicClassExists = true;
        } catch (ClassNotFoundException e) {
            // Ignore
        }

        try {
            Class.forName("org.apache.shardingsphere.driver.jdbc.core.datasource.ShardingSphereDataSource");
            shardingClassExists = true;
        } catch (ClassNotFoundException e) {
            // Ignore
        }

        HIKARI_MONITOR_EXISTS = hikariMonitorExists;
        SHARDING_CLASS_EXISTS = shardingClassExists;
        DYNAMIC_CLASS_EXISTS = dynamicClassExists;
    }

    public synchronized void startRun(ApplicationContext applicationContext) {
        if (started) {
            log.warn("duiba datasource monitor already started, please don't call start again");
            return;
        }
        Map<String, DataSource> beansOfType = applicationContext.getBeansOfType(DataSource.class);

        dataSourceMap = wrapperDatasourceMap(beansOfType);

        if (dataSourceMap.isEmpty()) {
            return;
        }

        singleThreadPool = new ThreadPoolExecutor(1, 1, 0L
                , TimeUnit.MILLISECONDS, new ArrayBlockingQueue<>(1)
                , new ThreadFactoryBuilder().setNamePrefix(THREAD_NAME_PREFIX).build());

        singleThreadPool.execute(this::scanDataSources);

        started = true;
    }

    /**
     * 包装真实的DataSource
     *
     * @param beansOfType 从上下文获取的DataSource
     * @return 真实的DataSource
     */
    private Map<String, DataSource> wrapperDatasourceMap(Map<String, DataSource> beansOfType) {
        Map<String, DataSource> dsMap = Maps.newHashMap();
        for (Map.Entry<String, DataSource> entry : beansOfType.entrySet()) {
            DataSource ds = entry.getValue();
            if (SHARDING_CLASS_EXISTS && ds instanceof ShardingSphereDataSource) {
                Map<String, ShardingSphereMetaData> metaDataMap = ((ShardingSphereDataSource) ds).getContextManager()
                        .getMetaDataContexts().getMetaDataMap();
                for (Map.Entry<String, ShardingSphereMetaData> shardingMetaDataEntry : metaDataMap.entrySet()) {
                    Map<String, DataSource> shardingMetaDataSource = shardingMetaDataEntry.getValue().getResource()
                            .getDataSources();
                    for (Map.Entry<String, DataSource> dataSourceEntry : shardingMetaDataSource.entrySet()) {
                        if (!dsMap.containsKey(dataSourceEntry.getKey())) {
                            dsMap.put(dataSourceEntry.getKey(), dataSourceEntry.getValue());
                        }
                    }
                }
            } else if (DYNAMIC_CLASS_EXISTS && ds instanceof DynamicRoutingDataSource) {
                Map<String, DataSource> dataSources = ((DynamicRoutingDataSource) ds).getDataSources();
                for (Map.Entry<String, DataSource> dynamicRoutingEntry : dataSources.entrySet()) {
                    if (dynamicRoutingEntry.getValue() instanceof ItemDataSource) {
                        if (!dsMap.containsKey(dynamicRoutingEntry.getKey())) {
                            dsMap.put(dynamicRoutingEntry.getKey(), ((ItemDataSource) dynamicRoutingEntry.getValue())
                                    .getDataSource());
                        }
                    }
                }
            }
        }
        return dsMap;
    }

    /**
     * 循环扫描数据源
     */
    private void scanDataSources() {
        do {
            try {
                for (Map.Entry<String, DataSource> entry : dataSourceMap.entrySet()) {
                    DataSource ds = entry.getValue();
                    scanDataSource(entry.getKey(), ds);
                }
                try {
                    TimeUnit.MILLISECONDS.sleep(1000L);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            } catch (Exception e) {
                log.warn(e.getMessage(), e);
            }
        } while (!Thread.currentThread().isInterrupted());
    }

    /**
     * 扫描数据源
     *
     * @param key 数据源ke
     * @param ds  数据源
     */
    private void scanDataSource(String key, DataSource ds) {
        if (isValid(ds)) {
            if (ds instanceof HikariDataSource) {
                DataSourceMonitorBO dataSourceMonitorBO = wrapperMonitorBO(ds);
                if (null != dataSourceMonitorBO) {
                    warnIfTooFull(key, dataSourceMonitorBO);
                }
            }
        }
    }

    /**
     * 告警处理
     *
     * @param key                 数据源key
     * @param dataSourceMonitorBO 连接池BO
     */
    private void warnIfTooFull(String key, DataSourceMonitorBO dataSourceMonitorBO) {
        // 池中有多少个空闲连接，它们可以被checkout
        int numIdleConnections = dataSourceMonitorBO.getNumAllocated() - dataSourceMonitorBO.getNumActive();

        if (dataSourceMonitorBO.getNumActive() >= THRESHOLD_VALUE
                && dataSourceMonitorBO.getNumAllocated() >= dataSourceMonitorBO.getMaxTotal() - THRESHOLD_VALUE
                && dataSourceMonitorBO.getNumActive() >= dataSourceMonitorBO.getNumAllocated() - THRESHOLD_VALUE) {
            log.error(String.format(FORMAT_ONE, key, dataSourceMonitorBO.getMaxTotal()
                    , dataSourceMonitorBO.getNumActive(), numIdleConnections));
        }
    }

    /**
     * 包装连接池BO
     *
     * @param ds 数据源
     * @return 连接池BO
     */
    private DataSourceMonitorBO wrapperMonitorBO(DataSource ds) {
        if (HIKARI_MONITOR_EXISTS && ds instanceof HikariDataSource) {
            HikariDataSource hikariDs = (HikariDataSource) ds;
            HikariPoolMXBean hikariBean = hikariDs.getHikariPoolMXBean();
            if (null != hikariBean) {
                // 池中有多少个空闲连接，它们可以被checkout
                int numIdleConnections = hikariBean.getIdleConnections();
                // 池中被checkout的连接有多少个
                int numActiveConnections = hikariBean.getActiveConnections();
                // 最大允许个数
                int maxTotal = ((HikariDataSource) ds).getMaximumPoolSize();
                int threadsAwaitingConnection = hikariBean.getThreadsAwaitingConnection();
                return new DataSourceMonitorBO(numActiveConnections
                        , numActiveConnections + numIdleConnections, maxTotal, threadsAwaitingConnection);
            }
        }
        return null;
    }

    /**
     * 判断数据源类型<br>
     * 目前只支持shardingSphere
     *
     * @param ds 数据源
     * @return true/false
     */
    private boolean isValid(DataSource ds) {
        if (ds != null) {
            return ds instanceof HikariDataSource && !((HikariDataSource) ds).isClosed();
        }
        return false;
    }

    /**
     * 连接池BO
     */
    @Data
    @AllArgsConstructor
    public static class DataSourceMonitorBO {
        private Integer numActive;
        private Integer numAllocated;
        private Integer maxTotal;
        private Integer threadsAwaitingConnection;
    }

    @Override
    public void destroy() {
        if (singleThreadPool != null) {
            ThreadPoolExecutor pool = (ThreadPoolExecutor) singleThreadPool;
            log.info("关闭dataSource监控线程池,taskCount:{},completedCount:{}", pool.getTaskCount()
                    , pool.getCompletedTaskCount());
            pool.shutdown();
        }
    }
}
