package cn.com.duiba.cloud.biz.tool.constants;

import lombok.AllArgsConstructor;
import lombok.Getter;

import java.time.Duration;
import java.util.stream.Stream;

/**
 * 有效时间解析常量
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 1.0
 * @date 2021/12/6 3:27 下午
 **/
public class ExpireConstant {
    public static final String DEFAULT = "DEFAULT";
    public static final String SECOND_10 = "SECOND_10";
    public static final String SECOND_30 = "SECOND_30";
    public static final String MINUTE_1 = "MINUTE_1";
    public static final String MINUTE_5 = "MINUTE_5";
    public static final String MINUTE_15 = "MINUTE_15";
    public static final String MINUTE_30 = "MINUTE_30";
    public static final String HOUR_1 = "HOUR_1";
    public static final String HOUR_3 = "HOUR_3";
    public static final String HOUR_12 = "HOUR_12";
    public static final String DAY_1 = "DAY_1";
    public static final String DAY_3 = "DAY_3";
    public static final String DAY_7 = "DAY_7";
    public static final String DAY_15 = "DAY_15";
    public static final String DAY_30 = "DAY_30";

    /**
     * 缓存过期时间枚举
     */
    @Getter
    @AllArgsConstructor
    public enum ExpireEnum {
        /**
         * 默认1小时
         */
        DEFAULT(Duration.ofHours(1)),

        /**
         * 10秒
         */
        SECOND_10(Duration.ofSeconds(10)),

        /**
         * 30秒
         */
        SECOND_30(Duration.ofSeconds(30)),

        /**
         * 1分钟
         */
        MINUTE_1(Duration.ofMinutes(1)),

        /**
         * 5分钟
         */
        MINUTE_5(Duration.ofMinutes(5)),

        /**
         * 15分钟
         */
        MINUTE_15(Duration.ofMinutes(15)),

        /**
         * 30分钟
         */
        MINUTE_30(Duration.ofMinutes(30)),

        /**
         * 1小时
         */
        HOUR_1(Duration.ofHours(1)),

        /**
         * 3小时
         */
        HOUR_3(Duration.ofHours(3)),

        /**
         * 12小时
         */
        HOUR_12(Duration.ofHours(12)),

        /**
         * 1天
         */
        DAY_1(Duration.ofDays(1)),

        /**
         * 3天
         */
        DAY_3(Duration.ofDays(3)),

        /**
         * 7天
         */
        DAY_7(Duration.ofDays(7)),

        /**
         * 15天
         */
        DAY_15(Duration.ofDays(15)),

        /**
         * 30天
         */
        DAY_30(Duration.ofDays(30)),
        ;

        /**
         * 过期时间
         */
        private Duration expire;

        /**
         * 根据有效期名称数组查询第一个匹配到的过期时间枚举
         *
         * @param nameStream cache中value以#分割后的数组流
         * @return 缓存过期时间枚举
         */
        public static ExpireEnum findByFirstNameFormat(Stream<String> nameStream) {
            return ExpireEnum.valueOf(nameStream
                    .filter(name -> Stream.of(values())
                            .anyMatch(e -> e.name().equalsIgnoreCase(name)))
                    .findFirst()
                    .orElse(DEFAULT.name())
            );
        }

        /**
         * 根据有效期名称数组查询第一个匹配到的过期时间枚举
         *
         * @param names cache中value以#分割后的数组
         * @return 缓存过期时间枚举
         */
        public static ExpireEnum findByFirstNameFormat(String... names) {
            return findByFirstNameFormat(Stream.of(names));
        }
    }
}