package cn.com.duiba.cloud.biz.tool.utils;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.config.YamlPropertiesFactoryBean;
import org.springframework.core.env.Environment;
import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.util.ResourceUtils;

import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Properties工具类， 可载入多个properties、yml文件，
 * 相同的属性在最后载入的文件中的值将会覆盖之前的值，
 * 取不到从System.getProperty()获取。
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 1.0
 * @date 2021/12/6 8:03 下午
 **/
@Slf4j
public class PropertiesUtil {
    /**
     * 默认加载的文件，可通过继承覆盖（若有相同Key，优先加载后面的）
     */
    public static final String[] DEFAULT_CONFIG_FILE = new String[]{
            "classpath:application.properties", "classpath:bootstrap.properties",
            "classpath:application.yml", "classpath:bootstrap.yml"
    };
    @Getter
    private final Set<String> configSet = CollUtil.newLinkedHashSet();
    @Getter
    private final Properties properties = new Properties();
    private static Environment environment;

    private static final ResourceLoader RESOURCE_LOADER;
    private static final ResourcePatternResolver RESOURCE_RESOLVER;

    static {
        RESOURCE_LOADER = new DefaultResourceLoader();
        RESOURCE_RESOLVER = new PathMatchingResourcePatternResolver(RESOURCE_LOADER);
    }

    private static final class PropertiesLoaderHolder {
        private static PropertiesUtil INSTANCE;

        static {
            reloadInstance();
        }

        public static void reloadInstance() {
            // 获取平台及模块相关的配置文件
            Set<String> configSet = CollUtil.newLinkedHashSet();
            Resource[] resources = getResources("classpath*:/config/module-*.*");
            for (Resource resource : resources) {
                configSet.add("classpath:config/" + resource.getFilename());
            }
            //configSet.add("classpath:config/jeesite.yml");
            // 获取全局设置默认的配置文件（以下是支持环境配置的属性文件）
            Set<String> set = CollUtil.newLinkedHashSet();
            Collections.addAll(set, DEFAULT_CONFIG_FILE);
            // 获取 spring.config.location 外部自定义的配置文件
            String customConfigs = System.getProperty("spring.config.location");
            if (StrUtil.isNotBlank(customConfigs)) {
                for (String customConfig : StrUtil.split(customConfigs, ",")) {
                    if (!customConfig.contains("$")) {
                        customConfig = org.springframework.util.StringUtils.cleanPath(customConfig);
                        if (!ResourceUtils.isUrl(customConfig)) {
                            customConfig = ResourceUtils.FILE_URL_PREFIX + customConfig;
                        }
                    }
                    set.add(customConfig);
                }
            }
            // 获取 spring.profiles.active 活动环境名称的配置文件
            String[] configFiles = set.toArray(new String[0]);
            String profiles = System.getProperty("spring.profiles.active");
            if (StrUtil.isBlank(profiles)) {
                PropertiesUtil propsTemp = new PropertiesUtil(configFiles);
                profiles = propsTemp.getProperty("spring.profiles.active");
            }
            for (String location : configFiles) {
                configSet.add(location);
                if (StrUtil.isNotBlank(profiles)) {
                    if (location.endsWith(".properties")) {
                        configSet.add(substringBeforeLast(location, ".properties")
                                + "-" + profiles + ".properties");
                    } else if (location.endsWith(".yml")) {
                        configSet.add(substringBeforeLast(location, ".yml")
                                + "-" + profiles + ".yml");
                    }
                }
            }
            configFiles = configSet.toArray(new String[0]);
            log.debug("Loading module config: {}", (Object) configFiles);
            INSTANCE = new PropertiesUtil(configFiles);
        }
    }

    /**
     * 载入多个文件，路径使用Spring Resource格式，相同的属性在最后载入的文件中的值将会覆盖之前的值。
     */
    public PropertiesUtil(String... configFiles) {
        for (String location : configFiles) {
            try {
                Resource resource = getResource(location);
                if (resource.exists()) {
                    if (location.endsWith(".properties")) {
                        try (InputStreamReader is = new InputStreamReader(resource.getInputStream(), StandardCharsets.UTF_8)) {
                            properties.load(is);
                            configSet.add(location);
                        } catch (IOException ex) {
                            log.error("Load " + location + " failure. ", ex);
                        }
                    } else if (location.endsWith(".yml")) {
                        YamlPropertiesFactoryBean bean = new YamlPropertiesFactoryBean();
                        bean.setResources(resource);
                        for (Map.Entry<Object, Object> entry : Objects.requireNonNull(bean.getObject()).entrySet()) {
                            properties.put(ObjectUtil.toString(entry.getKey()),
                                    ObjectUtil.toString(entry.getValue()));
                        }
                        configSet.add(location);
                    }
                }
            } catch (Exception e) {
                log.error("Load " + location + " failure. ", e);
            }
        }
    }

    /**
     * 当前类实例
     */
    public static PropertiesUtil getInstance() {
        return PropertiesLoaderHolder.INSTANCE;
    }

    /**
     * 重新加载实例（重新实例化，以重新加载属性文件数据）
     */
    public static void reloadInstance() {
        PropertiesLoaderHolder.reloadInstance();
    }

    /**
     * 正则表达式预编译
     */
    private static final Pattern P_1 = Pattern.compile("\\$\\{.*?}");

    /**
     * 获取属性值，取不到从System.getProperty()获取，都取不到返回null
     */
    public String getProperty(String key) {
        if (environment != null) {
            String value = environment.getProperty(key);
            if (value != null) {
                return value;
            }
        }
        String value = properties.getProperty(key);
        if (value != null) {
            Matcher m = P_1.matcher(value);
            while (m.find()) {
                String g = m.group();
                String childKey = g.replaceAll("\\$\\{|}", "");
                value = StrUtil.replace(value, g, getProperty(childKey));
            }
            return value;
        } else {
            return System.getProperty(key);
        }
    }

    /**
     * 取出String类型的Property，但以System的Property优先，如果都为null则返回defaultValue值
     */
    public String getProperty(String key, String defaultValue) {
        String value = getProperty(key);
        return value != null ? value : defaultValue;
    }

    /**
     * 设置环境属性
     *
     * @param environment env属性
     */
    public static void setEnvironment(Environment environment) {
        PropertiesUtil.environment = environment;
    }

    private static String substringBeforeLast(String str, String separator) {
        if (!StrUtil.isEmpty(str) && !StrUtil.isEmpty(separator)) {
            int pos = str.lastIndexOf(separator);
            return pos == -1 ? str : str.substring(0, pos);
        } else {
            return str;
        }
    }

    /**
     * 搜索资源文件
     *
     * @param locationPattern 路径正则
     * @return 资源列表
     */
    private static Resource[] getResources(String locationPattern) {
        try {
            return RESOURCE_RESOLVER.getResources(locationPattern);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 获取资源加载器（可读取jar内的文件）
     *
     * @param location 路径
     * @return 资源
     */
    private static Resource getResource(String location) {
        return RESOURCE_LOADER.getResource(location);
    }
}