package cn.com.duiba.cloud.biz.tool.config.cache;

import cn.com.duiba.cloud.biz.tool.constants.ExpireConstant;
import cn.hutool.core.collection.ListUtil;
import org.jetbrains.annotations.NotNull;
import org.springframework.data.redis.cache.RedisCache;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.cache.RedisCacheWriter;
import org.springframework.util.StringUtils;

import java.util.List;

/**
 * 将包含在cacheNames中的过期时间数据提取并配置，达到自定义每个key有效时间的目的。<br>
 * 设置有效期规则：cacheName + "#" + ExpireConstant常量中的有效期策略
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 1.0
 * @date 2021/12/6 3:15 下午
 * @since 0.0.1
 **/
public class RedisCacheManagerConfiguration extends RedisCacheManager {
    public RedisCacheManagerConfiguration(RedisCacheWriter cacheWriter, RedisCacheConfiguration defaultCacheConfiguration) {
        super(cacheWriter, defaultCacheConfiguration);
    }


    /**
     * 设置有效期规则：cacheName + "#" + ExpireConstant常量中的有效期策略
     *
     * @param name        实际缓存key + "#" + ExpireConstant常量中的有效期策略
     * @param cacheConfig 缓存配置
     * @return 自定义配置
     */
    @NotNull
    @Override
    protected RedisCache createRedisCache(@NotNull String name, RedisCacheConfiguration cacheConfig) {
        //用"#"将字符分割为String数组，取首个为CacheName，剩下数组匹配有效期策略，匹配不到的按默认缓存时间来
        List<String> nameList = ListUtil.toList(StringUtils.delimitedListToStringArray(name, "#"));
        //从枚举中获取第一个匹配到的有效期策略
        ExpireConstant.ExpireEnum ttl = ExpireConstant.ExpireEnum.findByFirstNameFormat(nameList.stream());
        //设置有效期
        cacheConfig = cacheConfig.entryTtl(ttl.getExpire());
        //剔除策略，重新拼接得到实际的cacheName
        nameList.remove(ttl.name());
        String cacheName = String.join("#", nameList);
        return super.createRedisCache(cacheName, cacheConfig);
    }
}