package cn.com.duiba.cloud.biz.tool.message;

import cn.hutool.core.collection.CollUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.context.support.ResourceBundleMessageSource;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

/**
 * 错误消息构建工厂
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @date 2021/12/29 12:10 下午
 * @since 0.0.1
 **/
@Slf4j
public class ErrorFactory {
    /**
     * 没有配置国际化资源的缓存
     */
    private static Set<String> noModuleCache = CollUtil.newHashSet();
    private static MessageSourceAccessor errorMessageSourceAccessor;

    /**
     * 初始化国际化资源文件
     *
     * @param isvModules 国际化资源modules
     */
    public static void initMessageSource(List<String> isvModules) {
        HashSet<String> baseNamesSet = CollUtil.newHashSet();
        if (!isvModules.isEmpty()) {
            baseNamesSet.addAll(isvModules);
        }
        String[] totalBaseNames = baseNamesSet.toArray(new String[0]);
        log.info("加载错误码国际化资源：{}", StringUtils.arrayToCommaDelimitedString(totalBaseNames));
        ResourceBundleMessageSource bundleMessageSource = new ResourceBundleMessageSource();
        bundleMessageSource.setBasenames(totalBaseNames);
        MessageSourceAccessor messageSourceAccessor = new MessageSourceAccessor(bundleMessageSource);
        setErrorMessageSourceAccessor(messageSourceAccessor);
    }

    /**
     * 转换错误消息为国际化
     *
     * @param errorMeta 错误元对象
     * @param locale    请求中的local
     * @param params    占位符参数
     * @return 转化后的错误消息
     */
    public static Error<String> getError(ErrorMeta errorMeta, Locale locale, Object... params) {
        Assert.notNull(locale, "未设置Locale");
        final String code = errorMeta.getCode();
        String errorMessage = getErrorMessage(errorMeta.getIsvModule() + code, locale, params);
        if (StringUtils.isEmpty(errorMessage)) {
            errorMessage = errorMeta.getMsg();
        }
        final String errorMsg = errorMessage;
        return new Error<String>() {
            @Override
            public String getMsg() {
                return errorMsg;
            }

            @Override
            public String getCode() {
                return code;
            }
        };
    }

    public static void setErrorMessageSourceAccessor(MessageSourceAccessor errorMessageSourceAccessor) {
        ErrorFactory.errorMessageSourceAccessor = errorMessageSourceAccessor;
    }

    public static String getErrorMessage(String module, Locale locale, Object... params) {
        if (noModuleCache.contains(module)) {
            return null;
        }
        try {
            return errorMessageSourceAccessor.getMessage(module, params, locale);
        } catch (Exception e) {
            noModuleCache.add(module);
            return null;
        }
    }
}