package cn.com.duiba.cloud.biz.tool.config.datasource;

import cn.com.duibaboot.ext.autoconfigure.perftest.datasource.PerfTestRoutingDataSource;
import cn.hutool.core.thread.ThreadFactoryBuilder;
import cn.hutool.core.util.StrUtil;
import com.zaxxer.hikari.HikariDataSource;
import com.zaxxer.hikari.HikariPoolMXBean;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.shardingsphere.driver.jdbc.core.datasource.ShardingSphereDataSource;
import org.apache.shardingsphere.infra.metadata.ShardingSphereMetaData;
import org.apache.shardingsphere.infra.state.StateType;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.ApplicationContext;

import javax.sql.DataSource;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 兑吧数据源监控<br>
 * 只针对shardingSphere，其余数据源还是从ext中监控
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @date 2022-04-25 20:34
 * @since 0.1.0
 **/
@Slf4j
public class DuibaDataSourceMonitor implements DisposableBean {
    private boolean started = false;

    private Map<String, DataSource> dataSourceMap;

    private ExecutorService singleThreadPool;

    private static final int THRESHOLD_VALUE = 3;

    private static final String THREAD_NAME_PREFIX = "duiba-datasource-monitor-thread-";

    private static final String FORMAT_ONE = "datasource [%s]'s connectionPool is too full,max:%d,busy:%d,idle:%d";
    private static final String FORMAT_TWO = "datasource [%s]-[%s]'s connectionPool is too full,max:%d,busy:%d,idle:%d";

    /**
     * HikariDataSource.getHikariPoolMXBean方法是否存在，如果不存在，代表hikari版本较低,无法监控连接池使用情况
     */
    private static final boolean HIKARI_MONITOR_EXISTS;
    private static final boolean SHARDING_CLASS_EXISTS;

    static {
        boolean hikariMonitorExists = false;
        boolean shardingClassExists = false;
        try {
            HikariDataSource.class.getDeclaredMethod("getHikariPoolMXBean");
            hikariMonitorExists = true;
        } catch (NoSuchMethodException e) {
            log.warn("请把hikari升级到最新版，否则无法监控连接池使用数据");
        }

        try {
            Class.forName("org.apache.shardingsphere.driver.jdbc.core.datasource.ShardingSphereDataSource");
            shardingClassExists = true;
        } catch (ClassNotFoundException e) {
            // Ignore
        }

        HIKARI_MONITOR_EXISTS = hikariMonitorExists;
        SHARDING_CLASS_EXISTS = shardingClassExists;
    }

    public synchronized void startRun(ApplicationContext applicationContext) {
        if (started) {
            log.warn("duiba datasource monitor already started, please don't call start again");
            return;
        }
        dataSourceMap = applicationContext.getBeansOfType(DataSource.class);

        if (dataSourceMap.isEmpty()) {
            return;
        }

        singleThreadPool = new ThreadPoolExecutor(1, 1, 0L
                , TimeUnit.MILLISECONDS, new ArrayBlockingQueue<>(1)
                , new ThreadFactoryBuilder().setNamePrefix(THREAD_NAME_PREFIX).build());

        singleThreadPool.execute(this::scanDataSources);

        started = true;
    }

    /**
     * 循环扫描数据源
     */
    private void scanDataSources() {
        do {
            try {
                for (Map.Entry<String, DataSource> entry : dataSourceMap.entrySet()) {
                    DataSource ds = entry.getValue();
                    if (!(ds instanceof PerfTestRoutingDataSource)) {
                        scanDataSource(entry.getKey(), ds);
                    }
                }
                try {
                    TimeUnit.MILLISECONDS.sleep(1000L);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            } catch (Exception e) {
                log.warn(e.getMessage(), e);
            }
        } while (!Thread.currentThread().isInterrupted());
    }

    /**
     * 扫描数据源
     *
     * @param key 数据源ke
     * @param ds  数据源
     */
    private synchronized void scanDataSource(String key, DataSource ds) {
        if (isValid(ds)) {
            if (SHARDING_CLASS_EXISTS && ds instanceof ShardingSphereDataSource) {
                Map<String, ShardingSphereMetaData> metaDataMap = ((ShardingSphereDataSource) ds).getContextManager()
                        .getMetaDataContexts().getMetaDataMap();
                for (Map.Entry<String, ShardingSphereMetaData> entry : metaDataMap.entrySet()) {
                    Map<String, DataSource> dataSources = entry.getValue().getResource().getDataSources();
                    for (Map.Entry<String, DataSource> actualEntry : dataSources.entrySet()) {
                        if (actualEntry.getValue() instanceof HikariDataSource) {
                            DatasourceMonitorBO datasourceMonitorBO = wrapperMonitorBO(actualEntry.getValue());
                            if (null != datasourceMonitorBO) {
                                warnIfTooFull(key, actualEntry.getKey(), datasourceMonitorBO);
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * 告警处理
     *
     * @param key                 数据源key
     * @param actualDatasourceKey meta数据源key
     * @param datasourceMonitorBO 连接池BO
     */
    private void warnIfTooFull(String key, String actualDatasourceKey, DatasourceMonitorBO datasourceMonitorBO) {
        // 池中有多少个空闲连接，它们可以被checkout
        int numIdleConnections = datasourceMonitorBO.getNumAllocated() - datasourceMonitorBO.getNumActive();

        if (datasourceMonitorBO.getNumActive() >= THRESHOLD_VALUE
                && datasourceMonitorBO.getNumAllocated() >= datasourceMonitorBO.getMaxTotal() - THRESHOLD_VALUE
                && datasourceMonitorBO.getNumActive() >= datasourceMonitorBO.getNumAllocated() - THRESHOLD_VALUE) {
            log.error(String.format(FORMAT_TWO, key, actualDatasourceKey, datasourceMonitorBO.getMaxTotal()
                    , datasourceMonitorBO.getNumActive(), numIdleConnections));
            if (StrUtil.isBlank(actualDatasourceKey)) {
                log.error(String.format(FORMAT_ONE, key, datasourceMonitorBO.getMaxTotal()
                        , datasourceMonitorBO.getNumActive(), numIdleConnections));
            } else {
                log.error(String.format(FORMAT_TWO, key, actualDatasourceKey, datasourceMonitorBO.getMaxTotal()
                        , datasourceMonitorBO.getNumActive(), numIdleConnections));
            }
        }
    }

    /**
     * 包装连接池BO
     *
     * @param ds 数据源
     * @return 连接池BO
     */
    private DatasourceMonitorBO wrapperMonitorBO(DataSource ds) {
        if (HIKARI_MONITOR_EXISTS && ds instanceof HikariDataSource) {
            HikariDataSource hikariDs = (HikariDataSource) ds;
            HikariPoolMXBean hikariBean = hikariDs.getHikariPoolMXBean();
            if (null != hikariBean) {
                // 池中有多少个空闲连接，它们可以被checkout
                int numIdleConnections = hikariBean.getIdleConnections();
                // 池中被checkout的连接有多少个
                int numActiveConnections = hikariBean.getActiveConnections();
                // 最大允许个数
                int maxTotal = ((HikariDataSource) ds).getMaximumPoolSize();
                int threadsAwaitingConnection = hikariBean.getThreadsAwaitingConnection();
                return new DatasourceMonitorBO(numActiveConnections
                        , numActiveConnections + numIdleConnections, maxTotal, threadsAwaitingConnection);
            }
        }
        return null;
    }

    /**
     * 判断数据源类型<br>
     * 目前只支持shardingSphere
     *
     * @param ds 数据源
     * @return true/false
     */
    private boolean isValid(DataSource ds) {
        if (ds != null) {
            return SHARDING_CLASS_EXISTS && ds instanceof ShardingSphereDataSource
                    && ((ShardingSphereDataSource) ds).getContextManager().getStateContext()
                    .getCurrentState() == StateType.OK;
        }
        return false;
    }

    /**
     * 连接池BO
     */
    @Data
    @AllArgsConstructor
    public static class DatasourceMonitorBO {
        private Integer numActive;
        private Integer numAllocated;
        private Integer maxTotal;
        private Integer threadsAwaitingConnection;
    }

    @Override
    public void destroy() {
        if (singleThreadPool != null) {
            ThreadPoolExecutor pool = (ThreadPoolExecutor) singleThreadPool;
            log.info("关闭dataSource监控线程池,taskCount:{},completedCount:{}", pool.getTaskCount()
                    , pool.getCompletedTaskCount());
            pool.shutdown();
        }
    }
}
