package cn.com.duiba.cloud.biz.tool.exception;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.biz.tool.message.BaseError;
import cn.com.duiba.wolf.entity.JsonResult;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.UndeclaredThrowableException;

/**
 * 全局异常处理器
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @date 2022/1/5 10:15 上午
 * @since 0.0.7
 **/
@Slf4j
@RestControllerAdvice
public class GlobalExceptionHandler {
    @ExceptionHandler(BizException.class)
    public JsonResult<Void> bizExceptionHandler(BizException e) {
        log.warn("BizException [{}]", e.getMessage(), e);
        return JsonResult.fail(e.getCode(), e.getMessage());
    }

    @ExceptionHandler(UndeclaredThrowableException.class)
    public JsonResult<Void> undeclaredExceptionHandler(UndeclaredThrowableException e, HttpServletRequest request) {
        if (e.getCause() instanceof BizException) {
            return bizExceptionHandler((BizException) e.getCause());
        } else {
            return exceptionHandler(e, request);
        }
    }

    @ExceptionHandler(value = {MethodArgumentNotValidException.class, BindException.class})
    public JsonResult<Void> methodArgumentNotValidExceptionHandler(Exception e) {
        BindingResult bindResult = null;
        if (e instanceof BindException) {
            bindResult = ((BindException) e).getBindingResult();
        } else if (e instanceof MethodArgumentNotValidException) {
            bindResult = ((MethodArgumentNotValidException) e).getBindingResult();
        }
        String message = "参数错误";
        if (bindResult != null && bindResult.hasErrors()) {
            ObjectError objectError = bindResult.getAllErrors().get(0);
            message = objectError.getDefaultMessage();
        }
        return JsonResult.fail(BaseError.PARAM_ERROR.getCode(), message);
    }

    @ExceptionHandler({IllegalArgumentException.class})
    public JsonResult<Void> illegalArgumentExceptionHandler(IllegalArgumentException e) {
        log.error("IllegalArgumentException", e);
        return JsonResult.fail(e.getMessage());
    }

    @ExceptionHandler(Exception.class)
    public JsonResult<Void> exceptionHandler(Exception e, HttpServletRequest request) {
        String message = e.getMessage();
        if (e instanceof InvocationTargetException && message == null) {
            message = ((InvocationTargetException) e).getTargetException().getMessage();
        }
        log.error("Exception [{} -> {}]:", request.getRequestURI(), message, e);
        return JsonResult.fail(BaseError.SYSTEM_ERROR.getMsg());
    }
}