# 开放平台提供对外接口

## 1、接入开放平台

### 1.1 引入包

在**api层**引入

```
provided "cn.com.duiba.cloud:duiba-openapi-service-api:0.0.7"
compile "cn.com.duiba.cloud:duiba-openapi-service-base:0.0.7"
```
api包提供开放相关能力，不能对外打包暴露；
base包提供最基础的东西，可以对外打包暴露；

### 1.2 添加配置

```
openapi.enable=true
```

### 1.3 添加路由接口

**在api层添加路由接口，建议 新建包 openapi，将所有对外开放的接口放到该包下**

**最好对外开放的接口都重新封装一下放到openapi包下，避免与原有通用接口混淆**

```java
/**
 * 开放平台路由接口
 * @author huangguosheng@duiba.com.cn
 * @date 2022/1/20 3:57 下午
 **/
@AdvancedFeignClient("服务名")
public interface OpenApiRouteService {
    /**
     * api路由
     * @param param
     * @return
     * @throws Exception
     */
    Object route(JSONObject param) throws Exception;
}

```

实现路由接口，OpenApiRoute由openapi包提供，注入使用即可

```java

@RestController
public class OpenApiRouteServiceImpl implements OpenApiRouteService {

    @Autowired
    private OpenApiRoute openApiRoute;

    @Override
    public Object route(JSONObject param) throws Exception {
        return openApiRoute.apiRoute(param);
    }
}
```

### 1.4 添加路由接口配置

配置为上面的接口类+方法，用`#`相连

```java
openapi.route-class =cn.com.duiba.cloud.duiba.payment.service.api.remoteservice.OpenApiRouteService#route
```

### 1.5 添加接口声明注解



@OpenApi ：该注解放在接口类上，用来识别该类对外提供开放接口；<br/>
@OpenPath：该注解放在方法上，用来配置接口 如下实例；<br/>
OpenModel：该注解放在实体对象上，用来方便生成文档；<br/>
OpenModelProperty：该注解放在实体对象属性上，用来方便生成文档；<br/>


```
@OpenApi(prefix = "/payment")
@AdvancedFeignClient("duiba-payment-service")
public interface RemoteAccountService {
    @OpenPath(apiPath = "/query/accountState",apiDesc = "查询用户账户状态信息",needTenant=true)
    AccountStateDTO queryAccountState(AccountStateQueryParam param);

    @OpenPath(apiPath = "/query/accountBill",apiDesc = "查询账户账单接口")
    PageResponse<PayBillDTO> queryAccountBill(PayBillPageQueryParam param);
}

```

### 1.6 启动结果

服务启动后，会有如下日志打印，表明正常启动

```
成功读取OpenApi路径：payment/query/accountState
OpenApi注册成功：{"apiDesc":"查询用户账户状态信息","apiPath":"payment/query/accountState","className":"cn.com.duiba.cloud.duiba.payment.service.api.remoteservice.RemoteAccountService","method":"queryAccountState","methodParamClass":"cn.com.duiba.cloud.duiba.payment.service.api.param.AccountStateQueryParam","routeClass":"cn.com.duiba.cloud.duiba.payment.service.api.remoteservice.OpenApiRouteService#route","serviceName":"duiba-payment-service"}
成功读取OpenApi路径：payment/query/accountBill
OpenApi注册成功：{"apiDesc":"查询账户账单接口","apiPath":"payment/query/accountBill","className":"cn.com.duiba.cloud.duiba.payment.service.api.remoteservice.RemoteAccountService","method":"queryAccountBill","methodParamClass":"cn.com.duiba.cloud.duiba.payment.service.api.param.PayBillPageQueryParam","routeClass":"cn.com.duiba.cloud.duiba.payment.service.api.remoteservice.OpenApiRouteService#route","serviceName":"duiba-payment-service"}
```

以上步骤会将接口注册到开放平台。

**如果需要将接口对外需要告知开放平台相关人员将接口开放状态变更。**

## 2、测试：



## 3、开放文档编写

参考
http://gitlab2.dui88.com/business_platform/duiba-openapi-web/tree/feature/20211220-openapi/apidoc

一期文档添加在该路径下，自建，以markdown格式输出

可以借助 `OpenApiDocUtil` 快速输出ApiDoc数据，再完善即可
@OpenModel、@OpenModelProperty 两注解打在请求参数和返回对象上，可以生成更友好的文档

```java
Method method=RemoteOrderWriteService.class.getMethod("createOrder",OrderCreateParam.class);
        OpenApiDocUtil.createApiDoc(method);
// 会在控制台打印
```

## 4、注意

> 建议底层服务在api层新增包，编写对应接口（包一下），后续需要根据应用维度进行接口隔离，提供不同的接口权限，或接口特殊处理等，最好不要和通用接口 混合在一，否则后续很难分离。
>
> 接口写完后需要告知开放平台进行接口升级
>
> 所有返回和参数，都需要包装层简单对象，不能直接用基本类型

## 5、获取appId

底层接口注意应用维度隔离，开放平台会在每个接口对象参数的属性中统一塞入apiContext对象属性，底层可以从apiContext得到appId进行接口隔离

需要底层自己定义ApiContext对象或者直接用JSONObject，属性有 appId[Long]、appName[String]、appType[Integer]

例如"

```java
@Override
public PageResponse<PayBillDTO> queryAccountBill(PayBillPageQueryParam param){
        JSONObject apiContext=param.getApiContext();
        Long appId=apiContext.getLong("appId");
        return accountService.queryAccountBill(param);
}
```

## 6、事件通知

事件通知，主要用于通知开发者，做出相应操作

事件传播类型包括：广播、定向

- 广播：对所有已开启相关通知的应用都会进行发送
- 定向：对特定已开启相关通知的应用都会进行发送

事件强弱性：强（仅定向）、弱

- 强事件：依赖于MQ的重试，直到成功为止；
- 弱事件：第一次发送失败后，会在十分钟内再次发现推送，依然失败则废弃，不再推送。

### 使用方法

```java
// 发送广播通知
JSONObject contentObj=new JSONObject();
contentObj.put("test","123");
String destination=OpenApiConstant.buildDestination(OpenEventTypeEnum.GOODS_CHANGE.name());
String content=PushEventMessageParam.createBroadcastEvent(OpenEventTypeEnum.GOODS_CHANGE,RandomUtil.randomNumbers(16),contentObj);
Message<String> message=MessageBuilder.withPayload(content).build();
rocketMQTemplate.syncSend(destination,message);

// 发送定向通知
JSONObject contentObj=new JSONObject();
contentObj.put("test","123");
String destination=OpenApiConstant.buildDestination(OpenEventTypeEnum.GOODS_CHANGE.name());
String content=PushEventMessageParam.createDirectionalEvent(OpenEventTypeEnum.GOODS_CHANGE,RandomUtil.randomNumbers(16),contentObj,OpenEventStrategyEnum.strong,123L);
Message<String> message=MessageBuilder.withPayload(content).build();
rocketMQTemplate.syncSend(destination,message);
```

**内容要以简单对象方式传递**

需要在OpenEventTypeEnum中增加枚举，需要在一下文档中登记，并告知@大圣
https://alidocs.dingtalk.com/i/team/3YxXA1vAq4ABLXNy/docs/3YxXAa00JPYJAXNy?corpId=dingdaf09daa5c31eb8c35c2f4657eb6378f&iframeQuery=sheet_range%253Dst-850803ed_16_5_1_1

需要完善开放文档：
http://gitlab2.dui88.com/business_platform/duiba-openapi-web/tree/feature/20211220-openapi/apidoc/%E4%BA%8B%E4%BB%B6%E6%96%87%E6%A1%A3

