package cn.com.duiba.cloud.manage.service.api.model.enums;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.biz.tool.utils.Conditions;
import cn.com.duiba.cloud.manage.service.api.utils.Code;
import cn.com.duiba.cloud.manage.service.api.utils.EnumUtil;

import javax.validation.constraints.NotNull;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Arrays;

/**
 * @program: duiba-manage-service
 * @description: 服务时长枚举
 * @author: 郭云龙
 * @create: 2022-03-24 15:32
 **/
public enum ServiceTimeEnum {

    /**
     *
     */
    ONE_YEAR(1L, "1年", UnitEnum.Y.name(), 1),
    SEVEN_DAYS(7L, "7天", UnitEnum.D.name(), 7),
    CUSTOMIZE(99L, "自定义"),
    ;


    enum UnitEnum {
        Y, M, D;
    }

    /**
     * id
     */
    @Code
    private Long serviceTimeId;

    /**
     * 应用类型
     */
    private String serviceTimeName;

    /**
     * 单位 Y-年 M-月 D-日
     */
    private String unit;

    /**
     * 时长
     */
    private Integer duration;

    ServiceTimeEnum(Long serviceTimeId, String serviceTimeName, String unit, Integer duration) {

        this.serviceTimeId = serviceTimeId;
        this.serviceTimeName = serviceTimeName;
        this.unit = unit;
        this.duration = duration;
    }

    ServiceTimeEnum(Long serviceTimeId, String serviceTimeName) {
        this.serviceTimeId = serviceTimeId;
        this.serviceTimeName = serviceTimeName;
    }

    public Long getServiceTimeId() {
        return serviceTimeId;
    }

    public String getServiceTimeName() {
        return serviceTimeName;
    }

    public String getUnit() {
        return unit;
    }

    public Integer getDuration() {
        return duration;
    }


    public static String getNameById(Long id){
        return Arrays.stream(ServiceTimeEnum.values())
                .filter(o -> id.equals(o.getServiceTimeId()))
                .findFirst().orElse(ServiceTimeEnum.CUSTOMIZE)
                .getServiceTimeName();
    }

    /**
     * 计算服务结束时间
     * @param startTime 开始时间
     * @param endTime 结束时间
     * @param serviceTypeId 服务时长枚举id
     * @param giftTime 赠送时长
     * @return 服务结束时间
     */
    public static LocalDateTime addTime(@NotNull Long serviceTypeId,
                                        @NotNull LocalDateTime startTime,
                                        LocalDateTime endTime,
                                        Long giftTime
                                        ) throws BizException {
        // 校验开始时间（必须是今天或今天之后的时间）
        checkStartTime(startTime);
        giftTime = giftTime == null ? 0 : giftTime;

        ServiceTimeEnum serviceTimeEnum = EnumUtil.getEnum(ServiceTimeEnum.class, serviceTypeId);
        Conditions.expectTrue(serviceTimeEnum != null, "不支持的服务时长类型");

        LocalDateTime result;
        switch (serviceTimeEnum) {
            case ONE_YEAR:
                result = startTime.plusYears(1).plusDays(giftTime);
                break;
            case SEVEN_DAYS:
                result = startTime.plusDays(7).plusDays(giftTime);
                break;
            case CUSTOMIZE:
                Conditions.expectTrue(endTime != null, "服务结束时间不能为空");
                

                result = endTime;
                break;
            default:
                throw new BizException("不支持的服务时长类型");
        }
        return result;
    }

    /**
     * 校验服务开始时间
     * @param startTime 开始时间
     * @throws BizException 服务开始时间异常
     */
    public static void checkStartTime(LocalDateTime startTime) throws BizException {
        LocalDateTime todayStartTime = LocalDateTime.of(LocalDate.now(), LocalTime.MIN);
        Conditions.expectTrue(startTime.isAfter(todayStartTime), "服务开始时间只能选择今天或今天之后的日期");
    }
}
