package cn.com.duiba.cloud.manage.service.autologin.sdk.interceptor;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.manage.service.autologin.sdk.constant.OpenApiSdkErrorCode;
import cn.com.duiba.cloud.manage.service.autologin.sdk.interceptor.annotation.CanAccess;
import cn.com.duiba.cloud.manage.service.autologin.sdk.utils.OpenApiRequestTool;
import cn.com.duiba.wolf.entity.JsonResult;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.resource.ResourceHttpRequestHandler;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Set;

/**
 * 开放平台拦截器
 *
 * @author jiangyesheng
 * @version 1.0
 * @date 2022/5/24
 */
@Slf4j
public class OpenApiInterceptor implements HandlerInterceptor {

    @Resource
    private OpenApiInterceptorHandlerManager manager;

    private static final Set<String> EXCLUDE_PATHS = Sets.newHashSet();

    static {
        EXCLUDE_PATHS.add("/favicon.ico");
        EXCLUDE_PATHS.add("/monitor/check");
        EXCLUDE_PATHS.add("/error");
    }

    @Override
    public boolean preHandle(@NotNull HttpServletRequest request, @NotNull HttpServletResponse response, @NotNull Object handler) throws Exception {
        //静态资源放行
        if (handler instanceof ResourceHttpRequestHandler) {
            return Boolean.TRUE;
        }
        // 设置线程本地变量
        OpenApiRequestTool.setRequestInThreadLocal(request, response);
        //放行特殊请求
        if (isSpecialRequest(request, handler)) {
            return Boolean.TRUE;
        }
        // 拦截处理
        try {
            if (!manager.doHandler(handler)) {
                // 拦截器拦截，无权限访问
                interceptResponse(response, OpenApiSdkErrorCode.NO_PERMISSION.getCode(), OpenApiSdkErrorCode.NO_PERMISSION.getMsg());
                return Boolean.FALSE;
            }
        } catch (BizException e) {
            log.warn("openapi interceptor BizException, errorCode:{}, errorMsg:{}", e.getCode(), e.getMessage());
            interceptResponse(response, e.getCode(), e.getMessage());
            return Boolean.FALSE;
        } catch (Exception e) {
            log.error("openapi interceptor unknown Exception, error:", e);
            interceptResponse(response, OpenApiSdkErrorCode.SYSTEM_ERROR.getCode(), OpenApiSdkErrorCode.SYSTEM_ERROR.getMsg());
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }

    @Override
    public void afterCompletion(@NotNull HttpServletRequest request, @NotNull HttpServletResponse response, @NotNull Object handler, Exception ex) throws Exception {
        OpenApiRequestTool.removeRequestInThreadLocal();
    }

    /**
     * 拦截器拦截后处理响应信息
     * @throws IOException
     */
    private void interceptResponse(HttpServletResponse response, String code, String message) throws IOException {
        if (OpenApiRequestTool.isAsynchronousRequests()) {
            response.setHeader("Content-Type", "application/json;charset=UTF-8");
            response.getWriter().write(JSONObject.toJSONString(JsonResult.fail(code, message)));
        } else {
            response.setHeader("Content-Type", "text/html;charset=UTF-8");
            response.getWriter().write(message);
        }

    }

    /**
     * 是否是特殊的请求
     * @return
     */
    private boolean isSpecialRequest(HttpServletRequest request, Object handler) {
        String url = request.getRequestURI();
        if (EXCLUDE_PATHS.contains(url)) {
            return true;
        }

        if (handler instanceof HandlerMethod) {
            HandlerMethod method = (HandlerMethod) handler;
            CanAccess canAccess = AnnotationUtils
                    .findAnnotation(method.getMethod(), CanAccess.class);
            if (canAccess != null) {
                return true;
            }
        }

        return Boolean.FALSE;
    }

}
