package cn.com.duiba.cloud.manage.service.sdk.service.impl;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.manage.service.sdk.enums.SdkRedisKeyEnum;
import cn.com.duiba.cloud.manage.service.sdk.model.dto.SdkStaffAuthorityDTO;
import cn.com.duiba.cloud.manage.service.sdk.model.dto.SdkTenantRightsDTO;
import cn.com.duiba.cloud.manage.service.sdk.model.param.RemoteQueryStaffAuthorityParam;
import cn.com.duiba.cloud.manage.service.sdk.model.param.RemoteQueryTenantAppRightsParam;
import cn.com.duiba.cloud.manage.service.sdk.remoteservice.RemoteMgtPermissionService;
import cn.com.duiba.cloud.manage.service.sdk.service.MgtPermissionService;
import cn.com.duiba.cloud.single.sign.on.contract.common.exception.SsoRunTimeException;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.utils.DateUtils;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 权益权限服务
 *
 * @author jiangyesheng
 * @version 1.0
 * @date 2021/12/20
 */
@Service
public class MgtPermissionServiceImpl implements MgtPermissionService {

    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;
    @Resource
    private RemoteMgtPermissionService remoteMgtPermissionService;

    /** 权限超时时间，默认一个小时 */
    @Value("${duiba.mgt.auth.expire_time_minute:60}")
    private int authExpireTime;

    @Override
    public SdkTenantRightsDTO getRightsByCached(Long tenantId, Long tenantAppId) {
        // 计算当前时间到24点的时间戳，即每天24点刷新
        Date now = new Date();
        Date dayEndTime = DateUtils.getDayEndTime(now);
        long timeout = dayEndTime.getTime() - now.getTime();
        // 从缓存中拿
        return advancedCacheClient.getWithCacheLoader(
                SdkRedisKeyEnum.getRedisKey(SdkRedisKeyEnum.TENANT_APP_RIGHTS_CACHE, tenantId, tenantAppId),
                (int) timeout,
                TimeUnit.MILLISECONDS,
                true,
                () -> {
                    RemoteQueryTenantAppRightsParam remoteParam = new RemoteQueryTenantAppRightsParam();
                    remoteParam.setTenantId(tenantId);
                    remoteParam.setTenantAppId(tenantAppId);
                    try {
                        return remoteMgtPermissionService.queryTenantAppRights(remoteParam);
                    } catch (BizException e) {
                        throw new SsoRunTimeException("获取租户权益异常", e);
                    }
                });
    }

    @Override
    public boolean removeRightsCache(Long tenantId, List<Long> tenantAppIds) {
        if (Objects.isNull(tenantId) || CollectionUtils.isEmpty(tenantAppIds)) {
            throw new IllegalArgumentException("必填参数为空");
        }
        List<String> redisKeyList = tenantAppIds.stream()
                .map(o -> SdkRedisKeyEnum.getRedisKey(SdkRedisKeyEnum.TENANT_APP_RIGHTS_CACHE, tenantId, o))
                .collect(Collectors.toList());
        return advancedCacheClient.remove(redisKeyList.toArray(new String[]{}));
    }

    @Override
    public SdkStaffAuthorityDTO getAuthorityByCached(Long tenantId, Long tenantAppId, Long staffId) {
        // 到12点过期了
        int expireTime = authExpireTime * 60 * 1000;
        Date now = new Date();
        Date dayEndTime = DateUtils.getDayEndTime(now);
        long remainTime = dayEndTime.getTime() - now.getTime();
        if (remainTime < expireTime) {
            expireTime = (int) remainTime;
        }
        return advancedCacheClient.getWithCacheLoader(
                SdkRedisKeyEnum.getRedisKey(SdkRedisKeyEnum.STAFF_APP_AUTHORITY_CACHE, tenantId, tenantAppId, staffId),
                expireTime,
                TimeUnit.MILLISECONDS,
                true,
                () -> {
                    RemoteQueryStaffAuthorityParam remoteParam = new RemoteQueryStaffAuthorityParam();
                    remoteParam.setTenantId(tenantId);
                    remoteParam.setTenantAppId(tenantAppId);
                    remoteParam.setStaffId(staffId);
                    try {
                        return remoteMgtPermissionService.queryStaffAuthority(remoteParam);
                    } catch (BizException e) {
                        throw new SsoRunTimeException("获取员工权限异常", e);
                    }
                });
    }

    @Override
    public boolean removeAuthorityCache(Long tenantId, Long tenantAppId, List<Long> staffIds) {
        if (Objects.isNull(tenantId) || Objects.isNull(tenantAppId) ||CollectionUtils.isEmpty(staffIds)) {
            throw new IllegalArgumentException("必填参数为空");
        }

        List<String> redisKeyList = staffIds.stream()
                .map(o -> SdkRedisKeyEnum.getRedisKey(SdkRedisKeyEnum.STAFF_APP_AUTHORITY_CACHE, tenantId, tenantAppId, o))
                .collect(Collectors.toList());

        return advancedCacheClient.remove(redisKeyList.toArray(new String[]{}));
    }
}
