package cn.com.duiba.cloud.manage.service.sdk.util;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.manage.service.api.model.enums.exception.ErrorCode;
import cn.com.duiba.cloud.manage.service.api.utils.AssertUtil;
import cn.com.duiba.cloud.manage.service.sdk.config.CookieConfig;
import cn.com.duiba.cloud.manage.service.sdk.model.dto.ParamHeaderDTO;
import cn.com.duiba.cloud.manage.service.sdk.model.dto.TokenCookieDTO;
import cn.com.duiba.cloud.single.sign.on.client.tool.SsoContext;
import cn.com.duiba.cloud.single.sign.on.contract.tool.CookieUtil;
import cn.com.duiba.cloud.single.sign.on.contract.tool.SsoRequestTool;
import cn.com.duiba.wolf.utils.BlowfishUtils;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * @author yuanzhixin
 * @version 1.0
 * @description
 * @date 2021/11/8 15:14
 */
@Component
@Slf4j
public class LoginUtil {

    private static final ThreadLocal<HttpRequestDto> LOCAL = new InheritableThreadLocal<>();

    /**
     * 获取租户应用id
     * @return 租户应用id
     * @throws BizException 未获取到选择的应用id
     */
    public static Long getTenantAppId() throws BizException {
        ParamHeaderDTO paramCookie = get().getParamHeader();
        AssertUtil.isTrue(paramCookie != null && paramCookie.getTenantAppId() != null, ErrorCode.APP_INFO_INVALID);
        return paramCookie.getTenantAppId();
    }

    /**
     * 获取租户id
     *
     * @return 租户id
     */
    public static Long getTenantId() throws BizException {
        TokenCookieDTO tokenCookieDTO = get().getTokenCookie();
        AssertUtil.isTrue(tokenCookieDTO != null && tokenCookieDTO.getTenantId() != null, ErrorCode.TENANT_NOT_NULL);
        return tokenCookieDTO.getTenantId();
    }

    /**
     * 获取innerUserId
     *
     * @return 租户innerUserId
     */
    public static Long getInnerUserId() throws BizException {
        TokenCookieDTO tokenCookieDTO = get().getTokenCookie();
        AssertUtil.isTrue(tokenCookieDTO != null && tokenCookieDTO.getInnerUserId() != null, ErrorCode.TENANT_NOT_NULL);
        return tokenCookieDTO.getInnerUserId();
    }

    /**
     * 获取staffId
     *
     * @return 租户id
     */
    public static Long getStaffId() throws BizException {
        TokenCookieDTO tokenCookieDTO = get().getTokenCookie();
        AssertUtil.isTrue(tokenCookieDTO != null && tokenCookieDTO.getStaffId() != null, ErrorCode.TENANT_NOT_NULL);
        return tokenCookieDTO.getStaffId();
    }

    /**
     * 设置request和response ，必须在获取用户信息之前调用（一般是在LoginFilter开始时）,内部会自动解析出用户信息
     */
    public static void setThreadLocally(HttpServletRequest request,
                                        HttpServletResponse response,
                                        String tku) {
        clearThreadLocally();//先清理
        Assert.notNull(request, "RequestUtils 注入 request 为空");
        Assert.notNull(response, "RequestUtils注入 response 为空");
        get().init(request, response, tku);
    }

    private static HttpRequestDto get() {
        HttpRequestDto httpRequestDto = LOCAL.get();
        if (null == httpRequestDto) {
            httpRequestDto = new HttpRequestDto();
            LOCAL.set(httpRequestDto);
        }
        return httpRequestDto;
    }

    /**
     * 清空线程本地变量,必须在处理完用户请求之后调用（一般是在LoginFilter调用链结束时）
     */
    public static void clearThreadLocally() {
        LOCAL.remove();
    }

    /**
     * 保存cookie
     * @param tenantId 租户id
     * @param innerUserId 用户id
     * @param staffId 员工id
     */
    public static void saveCookie(Long innerUserId, Long tenantId, Long staffId) {
        TokenCookieDTO tokenCookieDTO = new TokenCookieDTO();
        tokenCookieDTO.setInnerUserId(innerUserId);
        tokenCookieDTO.setTenantId(tenantId);
        tokenCookieDTO.setStaffId(staffId);
        tokenCookieDTO.setTime(System.currentTimeMillis());
        String cookie = BlowfishUtils.encryptBlowfish(JSONObject.toJSONString(tokenCookieDTO), CookieConfig.KEY_ENCRYPT);
        // 保存60天
        SsoRequestTool.addCookie(CookieUtil.createCookie(CookieConfig.COOKIE_NAME, cookie, 60 * 24 * 3600));
    }

    /**
     * 保存cookie
     */
    public static void updateCookieTime() {
        String manageTku = SsoRequestTool.getCookie(CookieConfig.COOKIE_NAME);
        // 续约60天
        SsoRequestTool.addCookie(CookieUtil.createCookie(CookieConfig.COOKIE_NAME, manageTku, 60 * 24 * 3600));
    }

    /**
     * 校验cookie中的userId和登录的userId是否一致
     * todo: 参考cms-activity-web登录，不都走cookie
     */
    public static void checkUserId() {
        String ssoId = SsoRequestTool.getRequest().getParameter("innerUserId");
        Long ssoUserId;
        if (StringUtils.isBlank(ssoId)) {
            ssoUserId = SsoContext.getLoginState().getUserId();
        } else {
            log.info("租户跳转应用参数 innerUserId: {}", ssoId);
            ssoUserId = Long.valueOf(ssoId);
        }
        Long userId = getCookieUserId();
        if (ObjectUtils.notEqual(ssoUserId, userId)) {
            setThreadLocally(SsoRequestTool.getRequest(), SsoRequestTool.getResponse(), null);
            saveCookie(ssoUserId, null, null);
        } else {
            setThreadLocally(SsoRequestTool.getRequest(), SsoRequestTool.getResponse(),
                    SsoRequestTool.getCookie(CookieConfig.COOKIE_NAME));
            updateCookieTime();
        }
    }

    /**
     * 获取cookie中的用户id
     * @return cookie中的用户id
     */
    public static Long getCookieUserId() {
        TokenCookieDTO tokenCookieDTO = parseCookie();
        if (tokenCookieDTO == null) {
            return null;
        }
        return tokenCookieDTO.getInnerUserId();
    }

    /**
     * 解析cookie
     * @return TokenCookieDTO
     */
    private static TokenCookieDTO parseCookie() {
        String manageTku = SsoRequestTool.getCookie(CookieConfig.COOKIE_NAME);
        return RequestUtils.getTokenCookie(manageTku);
    }

    /**
     * 获取租户id或者空
     * @return 租户id或者空
     */
    public static Long getTenantIdOrNull() {
        TokenCookieDTO tokenCookieDTO = get().getTokenCookie();
        if (tokenCookieDTO != null && tokenCookieDTO.getTenantId() != null) {
            return tokenCookieDTO.getTenantId();
        }
        return null;
    }

    /**
     * 获取员工id或者空
     * @return 员工id或者空
     */
    public static Long getStaffIdOrNull() {
        TokenCookieDTO tokenCookieDTO = get().getTokenCookie();
        if (tokenCookieDTO != null && tokenCookieDTO.getStaffId() != null) {
            return tokenCookieDTO.getStaffId();
        }
        return null;
    }

    /**
     * 获取租户应用id或者空
     * @return
     */
    public static Long getTenantAppIdOrNull() {
        ParamHeaderDTO paramCookie = get().getParamHeader();
        if (paramCookie != null && paramCookie.getTenantAppId() != null) {
            return paramCookie.getTenantAppId();
        }
        return null;
    }

    /**
     * 请求dto
     */
    private static class HttpRequestDto {
        /**
         * 请求
         */
        private HttpServletRequest request;

        /**
         * 响应
         */
        private HttpServletResponse response;

        /**
         * 用户登录Cookie信息
         */
        private TokenCookieDTO tokenCookie;

        /**
         * 请求参数信息 - 用于封装通用请求参数
         */
        private ParamHeaderDTO paramHeader;

        /**
         * 从request中恢复用户信息
         *
         * @param request  请求
         * @param response 响应
         */
        public void init(HttpServletRequest request, HttpServletResponse response, String tku) {
            this.request = request;
            this.response = response;
            if (StringUtils.isBlank(tku)) {
                tokenCookie = null;
            } else {
                tokenCookie = RequestUtils.getTokenCookie(tku);
            }
            if (tokenCookie == null) {
                tokenCookie = new TokenCookieDTO();
                tokenCookie.setInnerUserId(SsoContext.getLoginState().getUserId());
            }

            // 获取cookie参数
            paramHeader = RequestUtils.getCookieParam(request);
        }

        public HttpServletRequest getRequest() {
            return request;
        }

        public HttpServletResponse getResponse() {
            return response;
        }

        public TokenCookieDTO getTokenCookie() {
            return tokenCookie;
        }

        public ParamHeaderDTO getParamHeader() {
            return paramHeader;
        }
    }
}
