package cn.com.duiba.dayu.api.client;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import cn.com.duiba.dayu.api.dto.SceneConfig;
import cn.com.duiba.dayu.api.enums.ArgumentType;
import cn.com.duiba.dayu.api.remoteservice.RemoteDayuABService;
import cn.com.duiba.dayu.api.result.DayuResult;
import cn.com.duiba.wolf.dubbo.DubboResult;

/**
 * Created by lizhihui on 2017-08-29 09:28.
 */
public class DayuClient {

    private Logger                 logger              = LoggerFactory.getLogger(getClass());

    // 配置的缓存 场景id->场景配置
    private Map<Long, SceneConfig> sceneConfigCacheMap = new ConcurrentHashMap<>();

    // 每个场景的计数器集合 场景id->计数器
    private Map<Long, AtomicLong>  sceneCounterMap     = new ConcurrentHashMap<>();

    // 解析次数(到达解析次数后,会重新请求场景配置并缓存)
    private Long                   count               = 50000L;

    // 配置缓存时间(默认30分钟,单位:毫秒ms)
    private Long                   expireTime          = 30 * 60 * 1000L;

    // 定时任务执行器,作用:刷新场景配置缓存
    private Timer                  timer               = new Timer(true);

    @Resource
    private RemoteDayuABService    remoteDayuABService;

    public void setCount(Long count) {
        this.count = count;
    }

    public void setExpireTime(Long expireTime) {
        this.expireTime = expireTime;
    }

    /**
     * 单一layer流量分配策略调用方法 解析请求 这个方法最大的特点:如果这个方法的返回值是null,
     *
     * @param sceneCode 场景的code
     * @param param 请求的参数
     * @param type 请求的参数类型
     * @return 配置文件解析结果
     */
    public DayuResult handleRequest(String sceneCode, Object param, ArgumentType type) {
        Map<ArgumentType, Object> params = new EnumMap<>(ArgumentType.class);
        params.put(type, param);
        return this.handleRequest(sceneCode, params);
    }

    /**
     * 多layer流量分配策略调用方法
     *
     * @param sceneCode 场景的code
     * @param params 请求的参数
     * @return 配置文件解析结果
     */
    public DayuResult handleRequest(String sceneCode, Map<ArgumentType, Object> params) {
        // 检查参数
        this.checkParam(params);

        // 解析场景id
        Long sceneId = Long.parseLong(sceneCode.split(":")[1]);

        // 预处理场景请求
        this.preHandleRequest(sceneId);

        // 2.从缓存中获取配置
        SceneConfig sceneConfig = sceneConfigCacheMap.get(sceneId);

        // 如果配置不为空.则直接解析返回
        if (sceneConfig != null) {
            return SceneUtils.resolve(sceneConfig, params);
        } else {
            this.getSceneConfigAndPutItIntoCache(sceneId);
            return SceneUtils.resolve(sceneConfigCacheMap.get(sceneId), params);
        }
    }

    /**
     * 获取场景配置并缓存
     *
     * @param sceneId 场景ID
     */
    private void getSceneConfigAndPutItIntoCache(Long sceneId) {
        DubboResult<SceneConfig> sceneConfigResult = remoteDayuABService.getSceneConfig(sceneId);

        // 接口调用成功
        if (sceneConfigResult != null && sceneConfigResult.isSuccess()) {
            SceneConfig sceneConfig = sceneConfigResult.getResult();

            // 并且配置不为null,放入缓存
            Optional.ofNullable(sceneConfig).ifPresent(config -> sceneConfigCacheMap.put(sceneId, config));
        } else {
            logger.error("dayu server is abnormal, sceneId:{}", sceneId);
        }
    }

    /**
     * 开启定时更新场景配置定时器
     * 
     * @param sceneId 场景id
     */
    private void startUpdateSceneConfigSchedule(Long sceneId) {
        timer.schedule(new TimerTask() {

            @Override
            public void run() {
                try {
                    getSceneConfigAndPutItIntoCache(sceneId);
                } catch (Exception e) {
                    logger.error("定时更新配置失败");
                }
            }
            // 立即执行定时任务
        }, expireTime, expireTime);

    }

    /**
     * 预处理解析配置请求
     * 
     * @param sceneId 场景id
     */
    private void preHandleRequest(Long sceneId) {
        // 获取该场景的定时器,如果是第一次请求初始化定时器
        AtomicLong parseCount = sceneCounterMap.putIfAbsent(sceneId, new AtomicLong(1L));

        if (parseCount == null) {
            this.startUpdateSceneConfigSchedule(sceneId);
        } else if (parseCount.incrementAndGet() % count == 0) {
            // 如果请求达到了指定次数,就重新请求Dubbo接口获取最新的配置文件
            this.getSceneConfigAndPutItIntoCache(sceneId);
        }
    }

    /**
     * 校验参数
     *
     * @param params 参数
     */
    private void checkParam(Map<ArgumentType, Object> params) {
        if (params == null || params.isEmpty()) {
            logger.error("error! the params is empty");
            throw new NullPointerException("error! the params is empty");
        }
    }
}
