package cn.com.duiba.developer.center.api.utils;

import cn.com.duiba.developer.center.api.domain.enums.VipLimitTypeEnum;
import com.google.common.collect.Lists;

import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * Created by xiaoxuda on 2016/11/15.
 * 提供按照用户vip等级做功能过滤的能力,用户调用 RemoteVipLimitService 接口获取
 */
public class VipLimitFilter implements Serializable {
    private static final long serialVersionUID = 1L;

    /**
     * 用户等级
     **/
    private Integer vipLevel;

    /**
     * app维度vip等级限制开关
     **/
    private boolean isFilterOpen;

    /**
     * 默认无参构造函数
     **/
    public VipLimitFilter() {
        isFilterOpen = false;
        vipLevel = 0;
    }

    /**
     * @param vipLevel     用户vip等级
     * @param isFilterOpen app的vip限制功能打开标记
     */
    public VipLimitFilter(int vipLevel, boolean isFilterOpen) {
        this.vipLevel = vipLevel;
        this.isFilterOpen = isFilterOpen;
    }

    /**
     * 等级限制过滤，使用传入规则的方式，降低用户接入的代码耦合
     * @param collection
     * @param getter 信息读取规则
     * @param <T>
     */
    public <T> void vipLimitFilter(Collection<T> collection, VipLimitInfoGetter<T> getter){
        //未开启vip过滤功能，不做处理
        if (!this.isFilterOpen) {
            return;
        }

        Iterator<T> iterator = collection.iterator();
        while(iterator.hasNext()) {
            if (!canPassVipLimit(iterator.next(),getter)) {
                iterator.remove();
            }
        }
    }

    /**
     * 等级限制判断，使用传入规则的方式，降低用户接入的代码耦合
     * @param t
     * @param getter
     * @param <T>
     * @return
     */
    public <T> boolean canPassVipLimit(T t, VipLimitInfoGetter<T> getter){
        //未开启商品维度vip限制开关，直接返回true
        if (VipLimitTypeEnum.OFF_LEVEL.getCode().equals(getter.getVipLimitType(t))) {
            return true;
        }

        //取不到限制等级直接返回true
        if (getter.getVipLimits(t) == null) {
            return true;
        }

        //根据等级限制类型做判断
        long vipLevelLong = this.vipLevel.longValue();
        if (VipLimitTypeEnum.ABOVE_LEVEL.getCode().equals(getter.getVipLimitType(t))) {
            return vipLevelLong >= getter.getVipLimits(t);
        }
        if (VipLimitTypeEnum.ACCURATE_LEVEL.getCode().equals(getter.getVipLimitType(t))) {
            return isCanDo(getter.getVipLimits(t), vipLevelLong);
        }
        // 其他为异常情况
        return false;
    }

    //当为精确设置某个等级可兑时，判断某个等级是否可以兑换
    private boolean isCanDo(long vipLimits, long vipLevel) {
        long v = 1L << vipLevel;
        long ret = vipLimits & v;
        return ret != 0;
    }

}
