package cn.com.duiba.developer.center.api.aspectj;

import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.api.domain.dto.DeveloperDto;
import cn.com.duiba.developer.center.api.utils.CacheConstants;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.dubbo.DubboResult;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import java.util.concurrent.TimeUnit;

/**
 * Created by xiaoxuda on 2017/2/27.
 */
@Aspect
public class Aspect4RemoteAppService {
    private AdvancedCacheClient advancedCacheClient;

    /**
     * 本地缓存app信息，避免秒杀场景对redis的冲击，防止redis挂掉对业务的雪崩式影响
     */
    private static Cache<Long, AppSimpleDto> appCache = CacheBuilder.newBuilder()
            .expireAfterWrite(20, TimeUnit.SECONDS).build();
    /**
     *映射appKey到appId
     */
    private static Cache<String, Long> key2IdCache = CacheBuilder.newBuilder()
            .expireAfterWrite(6, TimeUnit.HOURS).build();
    /**
     * 本地缓存developer信息，避免秒杀场景对redis的冲击，防止redis挂掉对业务的雪崩式影响
     */
    private static Cache<Long, DeveloperDto> developerCache = CacheBuilder.newBuilder()
            .expireAfterWrite(20, TimeUnit.SECONDS).build();

    private static String getKeyByAppId(Long appId) {
        return CacheConstants.KEY_APP + appId;
    }

    private static String getKeyByDeveloperId(Long developerId){
        return CacheConstants.KEY_DEVELOPER + developerId;
    }

    private AppSimpleDto getApp(Long appId){
        AppSimpleDto result = appCache.getIfPresent(appId);
        if (result == null && advancedCacheClient != null) {
            result = advancedCacheClient.get(getKeyByAppId(appId));
            if(result != null){
                appCache.put(appId, result);
                key2IdCache.put(result.getAppKey(), appId);
            }
        }
        return result;
    }

    private AppSimpleDto getApp(String appKey){
        AppSimpleDto result = null;
        Long appId = key2IdCache.getIfPresent(appKey);
        if(appId != null) {
            result = getApp(appId);
        }
        return result;
    }

    private DeveloperDto getDeveloperById(Long developerId){
        DeveloperDto result = developerCache.getIfPresent(developerId);
        if (result == null && advancedCacheClient != null) {
            result = advancedCacheClient.get(getKeyByDeveloperId(developerId));
            if(result != null){
                developerCache.put(developerId, result);
            }
        }
        return result;
    }


    /**
     * 拦截对dubbo接口RemoteAppService的getSimpleApp方法的调用，利用缓存降低调用量
     *
     * @param proceedingJoinPoint
     * @return
     */
    @Around("execution(* cn.com.duiba.developer.center.api.remoteservice.RemoteAppService.getSimpleApp(..))")
    public Object aroundGetSimpleApp(ProceedingJoinPoint proceedingJoinPoint) throws Throwable {
        //首先查询缓存，查询不到结果再调用dubbo接口
        Long appId = (Long) proceedingJoinPoint.getArgs()[0];
        Object result = getApp(appId);
        if (result != null) {
            result = DubboResult.successResult(result);
        } else {
            result = proceedingJoinPoint.proceed();
        }
        return result;
    }

    /**
     * 拦截对dubbo接口RemoteAppService的getSimpleApp方法的调用，利用缓存降低调用量
     *
     * @param proceedingJoinPoint
     * @return
     */
    @Around("execution(* cn.com.duiba.developer.center.api.remoteservice.RemoteAppService.getAppByAppKey(..))")
    public Object aroundGetAppByAppKey(ProceedingJoinPoint proceedingJoinPoint) throws Throwable {
        //首先查询缓存，查询不到结果在调用dubbo接口
        String appKey = (String) proceedingJoinPoint.getArgs()[0];
        Object result = getApp(appKey);
        if (result != null) {
            result = DubboResult.successResult(result);
        } else {
            result = proceedingJoinPoint.proceed();
        }
        return result;
    }

    @Around("execution(* cn.com.duiba.developer.center.api.remoteservice.RemoteDeveloperService.getDeveloperById(..))")
    public Object aroundGetDeveloperById(ProceedingJoinPoint proceedingJoinPoint) throws Throwable {
        //首先查询缓存，查询不到结果再调用dubbo接口
        Long developerId = (Long) proceedingJoinPoint.getArgs()[0];
        Object result = getDeveloperById(developerId);
        if (result != null) {
            result = DubboResult.successResult(result);
        } else {
            result = proceedingJoinPoint.proceed();
        }
        return result;
    }

    public void setAdvancedCacheClient(AdvancedCacheClient advancedCacheClient) {
        this.advancedCacheClient = advancedCacheClient;
    }
}
