package cn.com.duiba.developer.center.biz.service.credits.impl;

import cn.com.duiba.developer.center.api.domain.dto.AppBudgetDto;
import cn.com.duiba.developer.center.biz.event.AppUpdateEvent;
import cn.com.duiba.developer.center.biz.exception.StatusException;
import cn.com.duiba.developer.center.biz.dao.app.AppBudgetDao;
import cn.com.duiba.developer.center.biz.factory.AppGroupFactory;
import cn.com.duiba.developer.center.biz.service.credits.AppBudgetService;
import cn.com.duiba.developer.center.common.constants.CacheConstants;
import cn.com.duiba.developer.center.common.constants.DsConstants;
import cn.com.duiba.developer.center.common.support.BizEventBus;
import cn.com.duiba.wolf.cache.CacheClient;
import com.google.common.eventbus.Subscribe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Objects;

/**
 * Created by liuyao on 16/8/2.
 */
@Service
public class AppBudgetServiceImpl implements AppBudgetService {

    private static final Logger log= LoggerFactory.getLogger(AppServiceImpl.class);
    @Resource
    private AppBudgetDao appBudgetDao;
    @Resource
    private CacheClient memcachedClient;
    @Autowired
    private AppGroupFactory factory;
    @Autowired
    private BizEventBus eventBus;

    @PostConstruct
    public void init(){
        eventBus.register(this);
        factory.registFactoryBean(this);
    }

    @Subscribe
    public void AppUpdateLister(AppUpdateEvent event){
        removeCacheById(event.getAppId());
    }

    @Override
    public void clearMonthBudget(Long appId) {
        appBudgetDao.clearMonthBudget(appId);
        removeCacheById(appId);
    }
    @Override
    public void cleanDayBudget(Long appId) {
        appBudgetDao.cleanDayBudget(appId);
        removeCacheById(appId);
    }
    @Override
    @Transactional(DsConstants.DATABASE_CREDITS)
    public void deductBudget(Long appId, Integer actualPrice) throws StatusException {
        if (actualPrice > 0)
        {
            AppBudgetDto capp = appBudgetDao.findBudgetForUpDate(appId);
            // 日预算
            if (capp.getDayBudget() != null)
            {
                int ret = appBudgetDao.deductBudgetDay(appId, actualPrice);
                if (ret == 0)
                {
                    throw new StatusException(StatusException.CodeDayBudgetNotEnough);
                }
            }
            // 月预算
            if (capp.getBudget() != null)
            {
                int rt = appBudgetDao.deductBudgetMonth(appId, actualPrice);
                if (rt == 0)
                {
                    throw new StatusException(StatusException.CodeMonthBudgetNotEnough);
                }
            }
        }
        removeCacheById(appId);
    }

    @Override
    public AppBudgetDto findAppBudgetForUpdate(Long appId) {
        return appBudgetDao.findBudgetForUpDate(appId);
    }

    @Override
    public boolean updateForDeveloperBack(AppBudgetDto budget) {

        AppBudgetDto capp = appBudgetDao.findBudgetForUpDate(budget.getAppId());

        BigDecimal dayBudgetBD = null;
        Integer dayBudgetInt = null;
        // 判断每日预算是否需要修改
        if (budget.getDayBudget() != null) {
            dayBudgetBD = new BigDecimal(budget.getDayBudget());
            dayBudgetInt = dayBudgetBD.divide(BigDecimal.ONE, 2, BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal(100.0)).intValue();

            // 判断需要修改后的每日预算和修改前的每日预算是否不同
            if (!dayBudgetInt.equals(capp.getDayBudget())) {
                // 同步每日预算余额
                Integer remain = dayBudgetInt - (capp.getDayBudget() == null ? 0 : capp.getDayBudget()) + (capp.getDayBudgetRemain() == null ? 0 : capp.getDayBudgetRemain());
                capp.setDayBudgetRemain(remain < 0 ? 0 : remain);
                capp.setDayBudget(dayBudgetInt);
            }
        } else {
            capp.setDayBudgetRemain(null);
            capp.setDayBudget(null);
        }

        BigDecimal monthBudgetBD = null;
        Integer monthBudgetInt = null;
        // 判断每月预算是否需要修改
        if (budget.getBudget() != null) {
            monthBudgetBD = new BigDecimal(budget.getBudget());
            monthBudgetInt = monthBudgetBD.divide(BigDecimal.ONE, 2, BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal(100.0)).intValue();

            // 判断需要修改后的每月预算和修改前的每月预算是否不同
            if (!monthBudgetInt.equals(capp.getBudget())) {
                // 同步每月预算余额
                Integer remain = monthBudgetInt - (capp.getBudget() == null ? 0 : capp.getBudget()) + (capp.getBudgetRemain() == null ? 0 : capp.getBudgetRemain());
                capp.setBudgetRemain(remain < 0 ? 0 : remain);
                capp.setBudget(monthBudgetInt);
            }
        } else {
            capp.setBudgetRemain(null);
            capp.setBudget(null);
        }

        BigDecimal auditPricePerConsumerBD = new BigDecimal(budget.getAuditPricePerConsumer() == null ? 0 : budget.getAuditPricePerConsumer());
        capp.setAuditPricePerConsumer(auditPricePerConsumerBD.divide(BigDecimal.ONE, 2, BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal(100.0)).intValue());
        appBudgetDao.updateForDeveloperBack(capp);
        removeCacheById(budget.getAppId());
        return true;
    }
    private void removeCacheById(Long id){
        memcachedClient.remove(getAppIdKey(id));
    }

    private String getAppIdKey(Long id){
        StringBuilder sb = new StringBuilder();
        sb.append(CacheConstants.KEY_APP_BUDGET_BY_APP_ID).append(id);
        return sb.toString();
    }

    @Override
    public AppBudgetDto getObject(Long appId){
        AppBudgetDto a = memcachedClient.get(getAppIdKey(appId));
        if(Objects.equals(null,a)) {
            a = appBudgetDao.find(appId);
            memcachedClient.set(getAppIdKey(appId), a, 300);
        }
        return a;
    }

    @Override
    public Class<AppBudgetDto> getObjectType() {
        return AppBudgetDto.class;
    }

}
