package cn.com.duiba.developer.center.common.dao;

import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;
import org.mybatis.spring.SqlSessionTemplate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.datasource.DelegatingDataSource;

import com.google.common.base.Objects;

import cn.com.duiba.wolf.spring.datasource.AutoRoutingDataSource;

/**
 * Created by liuyao on 16/7/15.
 */
public class BaseDao {
    private static final Logger log = LoggerFactory.getLogger(BaseDao.class);

    private static Map<DatabaseSchema,SqlSessionTemplate> sessionTemplateMap;

    protected DatabaseSchema databaseSchema;

    public void setDatabaseSchema(DatabaseSchema databaseSchema){
        this.databaseSchema = databaseSchema;
    }

    /**
     * 获取合适的sqlSessionTemplate
     * @param isForceUseMasterDataSource 是否强制使用master库,当为true时会强制使用master库,当为false时会判断当前是否正在事务中,如果是的话底层仍然会使用master库,如果不是的话会根据配置的规则按权重走主库或slave只读库.建议只在执行增删改sql时传入此参数为true.类似select * for update的语句实际上只在事务中有用,故如果不走事务,这样的语句有可能会被分配到只读库.所以写代码时需要注意for update语句一定要放在事务中.
     * @return
     */
    private SqlSessionTemplate getSqlSessionTemplate(boolean isForceUseMasterDataSource){
        DatabaseSchema databaseSchema = chooseSchema();
        if(databaseSchema == null){
            throw new NullPointerException("please set databaseSchema in class:" + this.getClass().getName());
        }
        SqlSessionTemplate sqlSessionTemplate = sessionTemplateMap.get(databaseSchema);
        if(Objects.equal(null,sqlSessionTemplate)){
            throw new NullPointerException("当你看到这个报错的时候,你肯定没有配置返回恰当的sqlSessionTemplate,麻烦屈身在spring-datasource.xml配置一下吧");
        }

        DataSource ds = sqlSessionTemplate.getSqlSessionFactory().getConfiguration().getEnvironment().getDataSource();
        // DataSource ds = sqlMapClientWithAutoRoutingDataSource.getDataSource();
        if (ds instanceof DelegatingDataSource) {
            ds = ((DelegatingDataSource) ds).getTargetDataSource();
        }

        if(ds instanceof AutoRoutingDataSource){
            AutoRoutingDataSource autoRoutingDataSource = (AutoRoutingDataSource)ds;
            autoRoutingDataSource.determineCurrentLookupKeyByItSelf(isForceUseMasterDataSource);
        }
        return sqlSessionTemplate;
    }

    /**
     * 这个方法延迟到子类实现,由子类决定注入哪个sqlSessionTemplate,以使用不同的库
     * @return
     */
    private DatabaseSchema chooseSchema(){
        if(Objects.equal(null,this.databaseSchema)){
            return DatabaseSchema.CREDITS;//默认为主库
        }else{
            return this.databaseSchema;
        }
    };

    public int insert(String statement) {
        return getSqlSessionTemplate(true).insert(addNameSpace(statement));
    }

    public int delete(String statement) {
        return getSqlSessionTemplate(true).delete(addNameSpace(statement));
    }

    public int update(String statement) {
        return getSqlSessionTemplate(true).update(addNameSpace(statement));
    }

    public int delete(String statement, Object parameter) {
        return getSqlSessionTemplate(true).delete(addNameSpace(statement), parameter);
    }

    public <E> List<E> selectList(String statement, Object parameter, RowBounds rowBounds) {
        return getSqlSessionTemplate(false).selectList(addNameSpace(statement), parameter, rowBounds);
    }

    public void select(String statement, ResultHandler handler) {
        getSqlSessionTemplate(false).select(addNameSpace(statement), handler);
    }

    public <T> T selectOne(String statement, Object parameter) {
        return getSqlSessionTemplate(false).selectOne(addNameSpace(statement), parameter);
    }

    public <K, V> Map<K, V> selectMap(String statement, Object parameter, String mapKey) {
        return getSqlSessionTemplate(false).selectMap(addNameSpace(statement), parameter, mapKey);
    }

    public int insert(String statement, Object parameter) {
        return getSqlSessionTemplate(true).insert(addNameSpace(statement), parameter);
    }

    public <K, V> Map<K, V> selectMap(String statement, Object parameter, String mapKey, RowBounds rowBounds) {
        return getSqlSessionTemplate(false).selectMap(addNameSpace(statement), parameter, mapKey, rowBounds);
    }

    public void select(String statement, Object parameter, RowBounds rowBounds, ResultHandler handler) {
        getSqlSessionTemplate(false).select(addNameSpace(statement), parameter, rowBounds, handler);
    }

    public <E> List<E> selectList(String statement) {
        return getSqlSessionTemplate(false).selectList(addNameSpace(statement));
    }

    public void select(String statement, Object parameter, ResultHandler handler) {
        getSqlSessionTemplate(false).select(addNameSpace(statement), parameter, handler);
    }

    public <K, V> Map<K, V> selectMap(String statement, String mapKey) {
        return getSqlSessionTemplate(false).selectMap(addNameSpace(statement), mapKey);
    }

    public int update(String statement, Object parameter) {
        return getSqlSessionTemplate(true).update(addNameSpace(statement), parameter);
    }

    public <T> T selectOne(String statement) {
        return getSqlSessionTemplate(false).selectOne(addNameSpace(statement));
    }

    public <E> List<E> selectList(String statement, Object parameter) {
        return getSqlSessionTemplate(false).selectList(addNameSpace(statement), parameter);
    }

    /**
     * 获取mybatis命名空间
     * @param method
     * @return
     */
    protected String addNameSpace(String method){
        return getClass().getName()+"."+method;
    }

    public Map<DatabaseSchema, SqlSessionTemplate> getSessionTemplateMap() {
        return sessionTemplateMap;
    }

    public void setSessionTemplateMap(Map<DatabaseSchema, SqlSessionTemplate> sessionTemplateMap) {
        BaseDao.sessionTemplateMap = sessionTemplateMap;
    }

}
