package cn.com.duiba.tuia.dsp.engine.api.dsp;

import cn.com.duiba.spring.boot.starter.dsp.sampler.SamplerLog;
import cn.com.duiba.tuia.dsp.engine.api.config.DspRateLimiter;
import cn.com.duiba.tuia.dsp.engine.api.config.RateLimiterProperties;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.req.AdxCommonBidRequest;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.req.DspInfo;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.resp.AdxCommonBidResponse;
import cn.com.duiba.tuia.dsp.engine.api.enums.DspEnum;
import cn.com.duiba.tuia.dsp.engine.api.exception.DspException;
import cn.com.duiba.tuia.dsp.engine.api.filter.AdvertConditionFilterHolder;
import com.alibaba.fastjson.JSON;
import com.dianping.cat.Cat;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

/**
 * 提供adx-web进行外部竞价的门面接口
 *
 * @author lizhiheng
 */
@SuppressWarnings({"rawtypes", "UnstableApiUsage"})
@Component
@Slf4j
public class DspFacade extends DspFacadeInterface implements ApplicationContextAware {

    private ApplicationContext applicationContext;

    @Resource
    ExecutorService outDspExecutorService;
    @Resource
    RateLimiterProperties rateLimiterProperties;
    @Resource
    DspRateLimiter rateLimiter;

    @Autowired
    private ExecutorService executorService;

    private final Map<Integer, AbstractDspCaller> dspCallerMap = new HashMap<>();

    @Autowired
    private AdvertConditionFilterHolder advertConditionFilterHolder;

    private static final Logger LOGGER = LoggerFactory.getLogger(DspFacade.class);

    @Override
    public AdxCommonBidResponse doInvoke(AdxCommonBidRequest adxCommonBidRequest, List<DspInfo> dspInfos) throws DspException {
        dspInfos = advertConditionFilterHolder.filter(dspInfos);
        List<AdxCommonBidResponse> adxCommonBidResponses = this.generateFutures(adxCommonBidRequest, dspInfos);
        return this.comparePrice(adxCommonBidResponses);
    }

    @Override
    public String priceEncryption(BigDecimal price, Integer dspId) {
        AbstractDspCaller caller = getDspByDspId(dspId);
        return caller.priceEncryption(price);
    }


    @Override
    public String replaceUrl(String price, String url, Integer dspId) {
        AbstractDspCaller caller = getDspByDspId(dspId);
        return caller.replaceUrl(price, url);
    }

    @Override
    public void doWinCallBack(String url, Integer dspId) {
        AbstractDspCaller caller = getDspByDspId(dspId);
        executorService.execute(() -> caller.doWinCallBack(url));
    }

    @Override
    public void doClickCallBack(String url, Integer dspId) {
        AbstractDspCaller caller = getDspByDspId(dspId);
        executorService.execute(() -> caller.doClickCallBack(url));
    }

    @Override
    public void doExposureCallBack(String url, Integer dspId) {
        AbstractDspCaller caller = getDspByDspId(dspId);
        executorService.execute(() -> caller.doExposureCallBack(url));
    }

    public List<AdxCommonBidResponse> generateFutures(AdxCommonBidRequest adxCommonBidRequest, List<DspInfo> dspInfos) {
        if (CollectionUtils.isEmpty(dspInfos)) {
            return null;
        }
        if (rateLimiter.getRateLimiter().tryAcquire(rateLimiterProperties.getRequireTimeout(), TimeUnit.MICROSECONDS)) {
            List<CompletableFuture<AdxCommonBidResponse>> futures = new ArrayList<>();
            dspInfos.forEach(dspInfo -> {
                AbstractDspCaller dspCaller = getDspByDspId(dspInfo.getDspId());
                CompletableFuture<AdxCommonBidResponse> adxCommonBidResponseCompletableFuture =
                        CompletableFuture.supplyAsync(() -> dspCaller.doBid(adxCommonBidRequest, dspInfo), outDspExecutorService);
                futures.add(adxCommonBidResponseCompletableFuture);
            });
            try {
                CompletableFuture.allOf(futures.toArray(new CompletableFuture[]{})).get(rateLimiterProperties.getFutureTimeOut(), TimeUnit.MILLISECONDS);
            } catch (TimeoutException e) {
                Cat.logMetricForCount("DSP并行请求总线执行超时");
            } catch (InterruptedException e) {
                Cat.logMetricForCount("DSP并行请求总线执行中断");
            } catch (Exception e) {
                Cat.logMetricForCount("DSP并行请求总线执行失败");
                LOGGER.warn("DSP并行请求总线执行失败", e);
            }

            return futures.stream().map(future -> {
                try {
                    AdxCommonBidResponse now = future.getNow(null);
                    if (now != null) {
                        return now;
                    }
                } catch (Exception e) {
                    Cat.logMetricForCount("DSP请求单个DSP失败");
                    LOGGER.warn("DSP请求单个DSP失败", e);
                }
                return null;
            }).collect(Collectors.toList());
        } else {
            Cat.logMetricForCount("DSP限流");
            // 采样日志
            int i = (int) (Math.random() * 10000);
            if (i <= 1) {
                LOGGER.info("DSP请求限流，超过阈值，{}个/s", rateLimiterProperties.getTokenPerSec());
            }
            return null;
        }

    }

    @Override
    public void setApplicationContext(@NotNull ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    public AbstractDspCaller getDspByDspId(Integer dspId) {
        AbstractDspCaller dspCaller = dspCallerMap.get(dspId);
        if (dspCaller != null) {
            return dspCaller;
        }
        dspCaller = applicationContext.getBean(DspEnum.getByDspId(dspId).getClazz());
        dspCallerMap.put(dspId, dspCaller);
        return dspCaller;
    }
}