package cn.com.duiba.tuia.dsp.engine.api.dsp.kuaishou;

import cn.com.duiba.spring.boot.starter.dsp.sampler.SamplerLog;
import cn.com.duiba.tuia.dsp.engine.api.dsp.AbstractDspCaller;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.req.AdxCommonBidRequest;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.req.DspInfo;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.resp.AdxCommonBidResponse;
import cn.com.duiba.tuia.dsp.engine.api.dsp.kuaishou.bean.KuaishouBidRequestOuterClass;
import cn.com.duiba.tuia.dsp.engine.api.dsp.kuaishou.bean.KuaishouBidResponseOuterClass;
import cn.com.duiba.tuia.dsp.engine.api.dsp.kuaishou.converter.KuaishouDspRequestConverter;
import cn.com.duiba.tuia.dsp.engine.api.dsp.kuaishou.converter.KuaishouDspResponseConverter;
import cn.com.duiba.tuia.dsp.engine.api.enums.DspEnum;
import cn.com.duiba.tuia.dsp.engine.api.exception.DspException;
import cn.com.duiba.tuia.dsp.engine.api.util.CloseableHttpClientUtil;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import cn.com.duibaboot.ext.autoconfigure.httpclient.ssre.CanAccessInsideNetwork;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Hex;
import org.springframework.http.*;
import org.springframework.stereotype.Component;
import org.springframework.web.client.ResourceAccessException;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;

import static cn.com.duiba.tuia.dsp.engine.api.exception.DspErrorCode.*;


/**
 * @author lizhiheng
 */
@Component
@Slf4j
public class KuaishouDspInvoker extends AbstractDspCaller<KuaishouBidResponseOuterClass.KuaishouBidResponse, KuaishouBidRequestOuterClass.KuaishouBidRequest> {

    @Resource
    private KuaishouDspRequestConverter kuaishouDspRequestConverter;

    @Resource
    private KuaishouDspResponseConverter kuaishouDspResponseConverter;

    @Resource
    private KuaishouDspProperties kuaishouDspProperties;

    @Resource
    private CloseableHttpClientUtil httpClientUtil;

    @Resource(name = "dspRestTemplate")
    @CanAccessInsideNetwork
    private RestTemplate restTemplate;

    @Override
    protected KuaishouBidResponseOuterClass.KuaishouBidResponse invokeDsp(KuaishouBidRequestOuterClass.KuaishouBidRequest rtbRequest) {
        if (rtbRequest == null) {
            return null;
        }
        Cat.logMetricForCount("快手DSP调用");
        try {
            return CatUtils.executeInCatTransaction(() -> this.doHttpInvoke(kuaishouDspProperties.getUrl(), rtbRequest), "invokeDSP", "kuaishou");
        } catch (ResourceAccessException e) {
            throw new DspException(DSP_HTTP_INVOKE_FAILED, this.getDsp(), e);
        } catch (Exception e) {
            throw new DspException(DSP_INVOKE_FAILED, this.getDsp(), e);
        } catch (Throwable e) {
            SamplerLog.warn("快手其他调用异常", e);
            return null;
        }
    }

    private KuaishouBidResponseOuterClass.KuaishouBidResponse doHttpInvoke(String url, KuaishouBidRequestOuterClass.KuaishouBidRequest rtbRequest) throws IOException {
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/x-protobuf");
        HttpEntity<byte[]> requestEntity = new HttpEntity<>(rtbRequest.toByteArray(), headers);
        ResponseEntity<org.springframework.core.io.Resource> resEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity, org.springframework.core.io.Resource.class);
        org.springframework.core.io.Resource resource = resEntity.getBody();
        if (resource == null) {
            return null;
        }
        HttpStatus statusCode = resEntity.getStatusCode();
        KuaishouBidResponseOuterClass.KuaishouBidResponse body = KuaishouBidResponseOuterClass.KuaishouBidResponse.parseFrom(resource.getInputStream());
        Transaction transaction = Cat.newTransaction("invokeDSP", "kuaishouReturn");
        transaction.setStatus("-1");
        try {
            if (statusCode.is2xxSuccessful()) {
                if (HttpStatus.OK.value() == statusCode.value()) {
                    Cat.logMetricForCount("快手DSP返回");
                    transaction.setStatus(Message.SUCCESS);
                    return body;
                } else {
                    SamplerLog.info("快手DSP无内容返回,pddRequest[{}],body[{}]", rtbRequest, body);
                    Cat.logMetricForCount("快手DSP返回2xx");
                    return null;
                }
            } else if (statusCode.is5xxServerError()) {
                Cat.logMetricForCount("快手DSP返回5xx");
                SamplerLog.info("快手DSP返回5xx,pddRequest[{}],body[{}]", rtbRequest, body);
                return null;
            } else {
                Cat.logMetricForCount("快手DSP返回其他状态码");
                SamplerLog.info("快手DSP返回其他状态码,pddRequest[{}],body[{}]", rtbRequest, body);
                return null;
            }
        } finally {
            transaction.complete();
        }
    }


    @Override
    public String priceEncryption(BigDecimal price) {
        try {
            String token = kuaishouDspProperties.getToken();
            byte[] raw = token.getBytes(StandardCharsets.UTF_8);
            SecretKeySpec keySpec = new SecretKeySpec(raw, "AES");
            Cipher cipher = Cipher.getInstance("AES/ECB/PKCS5Padding");
            cipher.init(Cipher.ENCRYPT_MODE, keySpec);
            byte[] encrypt = cipher.doFinal(price.toString().getBytes(StandardCharsets.UTF_8));
            return Hex.encodeHexString(encrypt);
        } catch (Exception e) {
            log.warn("KuaishouDspInvoker 价格加密失败");
            return null;
        }
    }


    @Override
    protected Integer getDspId() {
        return DspEnum.DSP_8.getDspId();
    }

    @Override
    protected DspEnum getDsp() {
        return DspEnum.DSP_8;
    }

    @Override
    public KuaishouBidRequestOuterClass.KuaishouBidRequest convertReq(AdxCommonBidRequest adxCommonBidRequest, DspInfo dspInfo) throws DspException {
        try {
            return kuaishouDspRequestConverter.reqConvert(adxCommonBidRequest, dspInfo);
        } catch (Exception e) {
            throw new DspException(REQ_PARAM_CONVERT_ERROR, e);
        }
    }

    @Override
    public AdxCommonBidResponse convertResp(KuaishouBidResponseOuterClass.KuaishouBidResponse rtbResponse) throws DspException {
        try {
            return kuaishouDspResponseConverter.respConvert(rtbResponse);
        } catch (Exception e) {
            throw new DspException(RESP_PARAM_CONVERT_ERROR, this.getDsp(), e);
        }
    }

    @Override
    protected String replaceUrl(String price, String url) {
        if (url.contains("${WIN_PRICE}") && price != null) {
            url = url.replace("${WIN_PRICE}", price);
        }
        return url;
    }

    @Override
    public void doWinCallBack(String url) {
        httpClientUtil.doGet(url);
    }

    @Override
    public void doClickCallBack(String url) {
        httpClientUtil.doGet(url);
    }

    @Override
    public void doExposureCallBack(String url) {
        httpClientUtil.doGet(url);
    }
}
