/*
 * Copyright (c) Huawei Technologies Co., Ltd. 2020-2020. All rights reserved.
 */
package cn.com.duiba.tuia.dsp.engine.api.dsp.jinghong;

import com.google.common.base.Charsets;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;


/**
 * adx api鉴权参数工具类
 *
 * @author w00502190
 * @since 2020-10-21
 */
public class DigestAuthUtil {
    private static final String[] HEX_DIGITS =
            {"0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "a", "b", "c", "d", "e", "f"};

    // usertype为1表示开发者
    private static final String USERTYPE = "1";

    /**
     * 获取鉴权参数
     *
     * @param uri uri
     * @param workKey workKey
     * @param appId appId
     * @param keyId keyId
     * @return str
     */
    public static String getPostReqDigestHeader(String uri, String workKey, String appId, String keyId) {
        Long nonce = System.currentTimeMillis();

        StringBuilder digestSb = new StringBuilder();
        digestSb.append(nonce)
                .append(":")
                .append("POST")
                .append(":")
                .append(uri);

        String key = appId + ":" + uri.substring(1) + ":" + workKey;
        String digest = encryptHMACStr(digestSb.toString(), key.getBytes(Charsets.UTF_8), "HmacSHA256");
        String authDigest = getAuthorzationRegexORData(appId, uri.substring(1), String.valueOf(nonce), digest,
                "HmacSHA256", keyId);
        return authDigest;
    }

    /**
     * HMAC加密
     *
     * @param data 需要加密的字符串
     * @param key 密钥
     * @return 字符串
     */
    public static String encryptHMACStr(String data, byte[] key, String algorithm) {
        if (null == data) {
            return null;
        }
        byte[] bytes = encryptHMAC(data.getBytes(org.apache.commons.codec.Charsets.UTF_8), key, algorithm);
        if (null == bytes) {
            return null;
        }

        return byteArrayToHexString(bytes);
    }

    /**
     * HMAC加密
     *
     * @param data byte[]
     * @param key byte[]
     * @param algorithm String
     * @return byte[]
     */
    public static byte[] encryptHMAC(byte[] data, byte[] key, String algorithm) {
        SecretKey secretKey = new SecretKeySpec(key, algorithm);
        byte[] bytes = null;

        Mac mac;
        try {
            mac = Mac.getInstance(secretKey.getAlgorithm());
            mac.init(secretKey);
            bytes = mac.doFinal(data);
        } catch (Exception e) {
        }

        return bytes;
    }

    /**
     * 转换字节数组为十六进制字符串
     *
     * @param bytes 字节数组
     * @return 十六进制字符串
     */
    private static String byteArrayToHexString(byte[] bytes) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < bytes.length; i++) {
            sb.append(byteToHexString(bytes[i]));
        }
        return sb.toString();
    }

    /**
     * 将一个字节转化成十六进制形式的字符串
     *
     * @param b 字节数组
     * @return 字符串
     */
    private static String byteToHexString(byte b) {
        int ret = b;
        if (ret < 0) {
            ret += 256;
        }
        int m = ret / 16;
        int n = ret % 16;
        return HEX_DIGITS[m] + HEX_DIGITS[n];
    }

    public static String getAuthorzationRegexORData(String appId, String realm, String nonce, String response,
                                             String algorithm, String keyId) {
        StringBuilder regex = new StringBuilder(150);
        regex.append("Digest ").append("username").append("=").append(appId).append(",")
                .append("realm").append("=").append(realm).append(",")
                .append("nonce").append("=").append(nonce).append(",")
                .append("response").append("=").append(response).append(",")
                .append("algorithm").append("=").append(algorithm).append(",")
                .append("usertype").append("=").append(USERTYPE).append(",")
                .append("keyid").append("=").append(keyId);

        return regex.toString();
    }

    /**
     * 示例用法
     */
    public static void main(String[] args) {
        // 示例参数（实际使用时需替换为真实值）
        // 鉴权秘钥传开发者拿到的Hex之后的字符串
        String signKey = "8a0558d88dd26c195e6de487ca794ee900e90614e5673a613e74150c504f5849";
        // 发布商Id
        String publishId = "1298627449743028224";
        // 向请求广告的url
        String realm = "/ppsadx/getResult";
        // keyid是分配密钥的标识，对于同一个调用方，每一个keyid会有一个唯一的密钥
        String keyId = "3";
        try {
            String del = getPostReqDigestHeader(realm, signKey, publishId, keyId);
            System.out.println(del);
        } catch (Exception e) {
            e.printStackTrace();
        }

    }
}
