/**
 * Copyright (c) 2018, duiba.com.cn All Rights Reserved.
 */
package cn.com.duiba.tuia.dsp.engine.api.util;

import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.core.SessionCallback;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.core.ZSetOperations;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * 描述: 所有缓存方法的进一步封装
 * 目的: 主要是方便统一管理，统一切换实例
 *
 * @version v1.0
 * @auther guopengfei@duiba.com.cn
 * @date 2018/7/26 10:45
 */
@Service
public class StringRedisHandler {

    /**
     * Redis实例
     */
    @Resource(name = "redis03StringRedisTemplate")
    private StringRedisTemplate stringRedisTemplate;

    public StringRedisTemplate getTemplateInstance() {
        return stringRedisTemplate;
    }

    public String get(String key) {
        return stringRedisTemplate.opsForValue().get(key);
    }

    public void set(String key, String value) {
        stringRedisTemplate.opsForValue().set(key, value);
    }

    public void set(String key, String value, long timeout, TimeUnit unit) {
        stringRedisTemplate.opsForValue().set(key, value, timeout, unit);
    }

    public void append(String key, String value) {
        stringRedisTemplate.opsForValue().append(key, value);
    }

    public Long increment(String key, long delta) {
        return stringRedisTemplate.opsForValue().increment(key, delta);
    }

    public Double increment(String key, double delta) {
        return stringRedisTemplate.opsForValue().increment(key, delta);
    }

    public Boolean expire(String key, final long timeout, final TimeUnit unit) {
        return stringRedisTemplate.expire(key, timeout, unit);
    }

    public Long getExpire(String key, final TimeUnit unit) {
        return stringRedisTemplate.getExpire(key, unit);
    }

    public void delete(String key) {
        stringRedisTemplate.delete(key);
    }

    public void delete(List<String> keyList) {
        stringRedisTemplate.delete(keyList);
    }

    public Boolean hasKey(String key) {
        return stringRedisTemplate.hasKey(key);
    }

    public Boolean expireAt(String key, Date date) {
        return stringRedisTemplate.expireAt(key, date);
    }

    public Boolean setIfAbsent(String key, String value) {
        return stringRedisTemplate.opsForValue().setIfAbsent(key, value);
    }


    //**** HashOperations
    public Object hoGet(String key, Object hashKey) {
        return stringRedisTemplate.opsForHash().get(key, hashKey);
    }

    public Long hoIncrement(String key, Object hashKey, long delta) {
        return stringRedisTemplate.opsForHash().increment(key, hashKey, delta);
    }

    public Map<Object, Object> hoEntries(String key) {
        return stringRedisTemplate.opsForHash().entries(key);
    }

    public void hoPut(String key, Object hashKey, String value) {
        stringRedisTemplate.opsForHash().put(key, hashKey, value);
    }

    public void hoPutAll(String key, Map<String, String> map) {
        stringRedisTemplate.opsForHash().putAll(key, map);
    }

    //**** ZSetOperations
    public Boolean zsoAdd(String key, String value, double score) {
        return stringRedisTemplate.opsForZSet().add(key, value, score);
    }

    public Set<String> zsoRange(String key, long start, long end) {
        return stringRedisTemplate.opsForZSet().range(key, start, end);
    }

    public Set<ZSetOperations.TypedTuple<String>> zsoRangeByScoreWithScores(String key, double min, double max) {
        return stringRedisTemplate.opsForZSet().rangeByScoreWithScores(key, min, max);
    }

    //**** SetOperations
    public Boolean soIsMember(String key, Object o) {
        return stringRedisTemplate.opsForSet().isMember(key, o);
    }

    public Set<String> soMembers(String key) {
        return stringRedisTemplate.opsForSet().members(key);
    }

    public Long soSize(String key) {
        return stringRedisTemplate.opsForSet().size(key);
    }

    public Long soRemove(String key, Object... values) {
        return stringRedisTemplate.opsForSet().remove(key, values);
    }

    public Long soAdd(String key, String... values) {
        return stringRedisTemplate.opsForSet().add(key, values);
    }

    //**** ListOperations
    public Long loRightPush(String key, String value) {
        return stringRedisTemplate.opsForList().rightPush(key, value);
    }

    public List<String> loRange(String key, long start, long end) {
        return stringRedisTemplate.opsForList().range(key, start, end);
    }

    public Long loLeftPush(String key, String value) {
        return stringRedisTemplate.opsForList().leftPush(key, value);
    }

    public void loTrim(String key, long start, long end) {
        stringRedisTemplate.opsForList().trim(key, start, end);
    }

    /**
     * 批量获取Map型数据
     *
     * @param keys
     * @return
     */
    public Map<String, Map<String, String>> pipelineHget(List<String> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return new HashMap<>();
        }
        final Map<String, Map<String, String>> valuesMap = new HashMap<>(keys.size());

        List<Object> pipResults = stringRedisTemplate.executePipelined(new SessionCallback<Object>() {
            @Override
            public <K, V> Object execute(RedisOperations<K, V> operations) throws DataAccessException {
                StringRedisTemplate template = (StringRedisTemplate) operations;
                for (String key : keys) {
                    template.opsForHash().entries(key);
                }
                return null;
            }
        });

        for (int index = 0; index < keys.size(); index++) {
            Object pipResult = pipResults.get(index);
            String key = keys.get(index);
            if (pipResult != null) {
                Map valMap = (Map) pipResult;
                Map<String, String> hMap = new HashMap<>();
                valMap.forEach((vkey, val) -> {
                    if (vkey != null && val != null) {
                        hMap.put(vkey.toString(), val.toString());
                    }
                });

                valuesMap.put(key, hMap);
            }
        }

        return valuesMap;
    }

    /**
     * pipeline For String
     *
     * @param keys
     * @return
     */
    public Map<String, String> pipelineValue(List<String> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return new HashMap<>();
        }
        final Map<String, String> valueMap = new HashMap<>(keys.size());

        List<Object> pipResults = stringRedisTemplate.executePipelined(new SessionCallback<Object>() {
            @Override
            public <K, V> Object execute(RedisOperations<K, V> operations) throws DataAccessException {
                StringRedisTemplate template = (StringRedisTemplate) operations;
                for (String key : keys) {
                    template.opsForValue().get(key);
                }
                return null;
            }
        });

        for (int index = 0; index < keys.size(); index++) {
            Object pipResult = pipResults.get(index);
            String key = keys.get(index);
            if (pipResult != null) {
                valueMap.put(key, pipResult.toString());
            }
        }
        return valueMap;
    }

    public String hGetAll(String key) {
        Map<Object, Object> entries = stringRedisTemplate.opsForHash().entries(key);
        return JSONObject.toJSONString(entries);
    }
}
