package cn.com.duiba.api.enums;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang.StringUtils;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @Author: lufeng
 * @Description:活动工具的所有枚举类 整合duiba_hdtool和operation_activity
 * @Date: Created in 2018/11/6
 */
public enum ActivityToolTypeEnum {

    /*
     * 使用规则：
     * 查询大类，使用type来匹配会比较精确。
     * 查询小类，使用subType来匹配会比较精确
     * operatingType 对应入库活动表的type
     *
     * */

    //大类
    SHAKE("摇一摇", 1, "shake", "", 6),
    SCRATCHCARD("刮刮乐", 2, "scratchCard", "", 7),
    TURNTABLE("大转盘", 3, "turntable", "", 8),
    TIGER("摇奖机", 4, "tiger", "", 9),
    SLOTMACHINE("摇奖机", 4, "slotMachine", "", 9),
    TURNCARD("趣味翻牌", 5, "turnCard", "", 11),
    CUSTOM("自定义活动", 6, "custom", "", 12),

    //小类(自定义活动)
    NIUDAN("扭蛋机", 6, "", "niudan", 12),
    SMASHG("砸彩蛋", 6, "", "smashg", 12),
    BLOWBALLOON("打气球", 6, "", "blowBalloon", 12),
    CATCHRABBIT("抓兔子", 6, "", "catchRabbit", 12),
    DIGGOLD("挖金矿", 6, "", "digGold", 12),
    NEWDOLLMAC("新版娃娃机", 6, "", "newDollMac", 12),
    ROLLDICE("摇骰子", 6, "", "rollDice", 12),
    FISHINGGOLD("大海捞金", 6, "", "fishinggold", 12),
    HOPSCOTCH("跳房子", 6, "", "hopscotch", 12),
    DOLLMAC("娃娃机", 6, "", "dollMac", 12),
    DRAWLOT("摇签", 6, "", "drawLot", 12),
    CUTROPE("割绳子", 6, "", "cutRope", 12),
    REDPACKET("扯红包", 6, "", "redPacket", 12),
    EXPRESS("拆快递", 6, "", "express", 12),
    LOTTERYMACHINE("彩票机", 6, "", "lotteryMachine", 12),
    COLORBALL("彩球机", 6, "", "colorball", 12),
    PINBALL("弹珠机", 6, "", "pinball", 12),
    KICKTREE("踹一踹", 6, "", "kickTree", 12),
    SETDOLL("套娃", 6, "", "setDoll", 12),
    CLCARD("集卡活动", 6, "", "clcard", 12),
    QUESTION("答题活动", 6, "", "question", 101);


    //活动名称
    private String name;
    //duiba_hdtool活动type
    private Integer type;
    //duiba_hdtool活动subType
    private String subType;
    //活动小类。当type=6时使用
    private String customSubType;
    //operating_activity入库活动type
    private Integer operatingType;

    private static ImmutableSet.Builder<ActivityToolTypeEnum> builder = ImmutableSet.builder();
    private static ImmutableMap.Builder<String, ActivityToolTypeEnum> subTypeMap = ImmutableMap.builder();


    ActivityToolTypeEnum(String name, Integer type, String subType, String customSubType, Integer operatingType) {
        this.name = name;
        this.type = type;
        this.subType = subType;
        this.customSubType = customSubType;
        this.operatingType = operatingType;

    }

    public String getName() {
        return name;
    }

    public Integer getType() {
        return type;
    }

    public String getSubType() {
        return subType;
    }

    public String getCustomSubType() {
        return customSubType;
    }

    public Integer getOperatingType() {
        return operatingType;
    }

    static {
        for (ActivityToolTypeEnum typeClass : ActivityToolTypeEnum.values()) {
            builder.add(typeClass);
            if (!StringUtils.isEmpty(typeClass.getSubType())) {
                subTypeMap.put(typeClass.getSubType(), typeClass);
            } else {
                subTypeMap.put(typeClass.getCustomSubType(), typeClass);
            }
        }
    }

    /**
     * 获取全部 枚举值和名称
     *
     * @return
     */
    public static ImmutableSet<ActivityToolTypeEnum> getAll() {
        return builder.build();
    }

    /**
     * 根据subType获取具体枚举类型
     *
     * @param subType
     * @return
     */
    public static ActivityToolTypeEnum getBySubType(String subType) {
        return subTypeMap.build().get(subType);
    }

    /**
     * 根据type获取枚举类型
     *
     * @param type
     * @return
     */
    public static List<ActivityToolTypeEnum> getByType(Integer type) {
        return builder.build().stream().filter(typeEnum -> typeEnum.getType().equals(type)).collect(Collectors.toList());
    }

    /**
     * 根据operatingType获取枚举类型
     *
     * @param operatingType
     * @return
     */
    public static List<ActivityToolTypeEnum> getByOperatingType(Integer operatingType) {
        return builder.build().stream().filter(typeEnum -> typeEnum.getOperatingType().equals(operatingType)).collect(Collectors.toList());
    }


}
