package cn.com.duiba.galaxy.sdk.base;

import cn.com.duiba.galaxy.sdk.message.BizError;
import lombok.Getter;

import javax.validation.constraints.NotNull;

/**
 * Controller响应json统一格式
 *
 * @author fangxiaorun
 * @date 2022/09/28
 */
@Getter
public class JsonResult<T> {
    /**
     * 默认成功编码
     */
    private static final String SUCCESS_CODE = "000000";

    /**
     * 默认成功描述
     */
    private static final String SUCCESS_DESC = "OK";

    /**
     * 是否成功
     */
    @NotNull
    private Boolean success;

    /**
     * 响应码
     */
    @NotNull
    private String code;

    /**
     * 描述信息
     */
    @NotNull
    private String desc;

    /**
     * 当前服务器时间戳
     */
    @NotNull
    private Long timestamp;

    /**
     * 响应数据
     */
    private T data;

    private JsonResult() {
    }

    private JsonResult(Boolean success, String code, String desc, T data) {
        this.success = success;
        this.desc = desc;
        this.data = data;
        this.code = code;
        this.timestamp = System.currentTimeMillis();
    }

    /**
     * 带默认成功码、不含返回数据
     *
     * @return {@link JsonResult}<{@link T}>
     */
    public static <T> JsonResult<T> success() {
        return success(null);
    }


    /**
     * 带默认成功码、含返回数据
     *
     * @param data 数据
     * @return {@link JsonResult}<{@link T}>
     */
    public static <T> JsonResult<T> success(T data) {
        return new JsonResult<>(true, SUCCESS_CODE, SUCCESS_DESC, data);
    }

    /**
     * 带自定义错误码、自定义描述信息
     *
     * @param code 错误码
     * @param desc 描述
     * @return {@link JsonResult}<{@link T}>
     */
    public static <T> JsonResult<T> fail(String code, String desc) {
        return new JsonResult<>(false, code, desc, null);
    }


    /**
     * 带自定义错误码、自定义描述信息
     *
     * @param bizError 业务错误
     * @return {@link JsonResult}<{@link T}>
     */
    public static <T> JsonResult<T> fail(BizError bizError) {
        return new JsonResult<>(false, bizError.getCode(), bizError.getDesc(), null);
    }
}
