package cn.com.duiba.galaxy.sdk.apiextra;

import cn.com.duiba.galaxy.sdk.Api;
import cn.com.duiba.galaxy.sdk.utils.ExpireTime;

import java.util.List;
import java.util.Map;

/**
 * ⚠️个人数据建议放在userData中
 * key-value存储功能API
 * 本API中所有使用到的数据将会自动根据项目进行隔离，即不同项目之间无法互相访问数据，使用者无需关心不同项目key重复的问题
 * 请尽量避免在本数据结构中存储大文本，如果需要存储大文本，请使用{@link QueryTable}
 *
 * @author ZhouFeng zhoufeng@duiba.com.cn
 * @version $Id: KeyValueApi.java , v 0.1 2020-05-07 6:00 下午 ZhouFeng Exp $
 */
public interface KeyValueApi extends Api {

    /**
     * 获取{@code String}类型的值
     *
     * @param key 键
     * @return 如果值存在且尚未过期，返回对应值；否则返回{@code null}
     */
    String get(String key);

    /**
     * 获取{@code Long}类型的值
     *
     * @param key 键
     * @return 如果值存在且尚未过期，返回对应值；否则返回{@code null}
     */
    Long getLong(String key);

    /**
     * 批量获取{@code String}类型的值，如果某个键不存在，将使用{@code null}填充value值
     *
     * @param keys 键组成的列表
     * @return 如果value值存在且尚未过期，目标位置为对应值；否则目标位置为{@code null}
     */
    Map<String, String> multiGet(List<String> keys);

    /**
     * 批量获取{@code Long}类型的值，如果某个键不存在，将使用{@code null}填充value值
     *
     * @param keys 键组成的列表
     * @return 如果value值存在且尚未过期，目标位置为对应值；否则目标位置为{@code null}
     */
    Map<String, Long> multiGetLong(List<String> keys);

    /**
     * 设置一个键值对，值为{@code String}类型，默认失效时间为 January 1, 2099, 00:00:00 GMT
     * 如果键不存在，则新增一条记录；如果键已经存在，则会修改值，但不会修改失效时间
     *
     * @param key   键
     * @param value 值
     */
    void set(String key, String value);

    /**
     * 设置一个键值对，值为{@code String}类型，并设置失效时间
     * 如果键不存在，则新增一条记录；如果键已经存在，则会修改值，并且设置失效时间
     *
     * @param key        键
     * @param value      值
     * @param expireTime 失效时间
     */
    void set(String key, String value, ExpireTime expireTime);

    /**
     * 设置一个键值对，值为{@code Long}类型，默认失效时间为 January 1, 2099, 00:00:00 GMT
     * 如果键不存在，则新增一条记录；如果键已经存在，则会修改值，但不会修改失效时间
     *
     * @param key   键
     * @param value 值
     */
    void set(String key, Long value);

    /**
     * 设置一个键值对，值为{@code Long}类型，并设置失效时间
     * 如果键不存在，则新增一条记录；如果键已经存在，则会修改值，并且设置失效时间
     *
     * @param key        键
     * @param value      值
     * @param expireTime 失效时间
     */
    void set(String key, Long value, ExpireTime expireTime);

    /**
     * 设置一个键值对，值为{@code String}类型，默认失效时间为 January 1, 2099, 00:00:00 GMT
     * 只有当键不存在时才会设置成功
     *
     * @param key   键
     * @param value 值
     * @return 设置记录成功时返回 {@code true};否则返回 {@code false}
     */
    boolean setIfAbsent(String key, String value);

    /**
     * 设置一个键值对，值为{@code String}类型，并设置失效时间
     * 只有当键不存在时才会设置成功
     *
     * @param key        键
     * @param value      值
     * @param expireTime 失效时间
     * @return 设置记录成功时返回 {@code true};否则返回 {@code false}
     */
    boolean setIfAbsent(String key, String value, ExpireTime expireTime);

    /**
     * 设置一个键值对，值为{@code Long}类型，默认失效时间为 January 1, 2099, 00:00:00 GMT
     * 只有当键不存在时才会设置成功
     *
     * @param key   键
     * @param value 值
     * @return 设置记录成功时返回 {@code true};否则返回 {@code false}
     */
    boolean setIfAbsent(String key, Long value);

    /**
     * 设置一个键值对，值为{@code Long}类型，并设置失效时间
     * 只有当键不存在时才会设置成功
     *
     * @param key        键
     * @param value      值
     * @param expireTime 失效时间
     * @return 设置记录成功时返回 {@code true};否则返回 {@code false}
     */
    boolean setIfAbsent(String key, Long value, ExpireTime expireTime);

    /**
     * 增加key对应longValue值，并设置失效时间
     * 如果key不存在，则insert一条longValue为0的数据，再增加offset； 否则在原值上进行增加offset，并修改失效时间
     *
     * @param key        键
     * @param offset     增量:  offset可以为负数，表示减少
     * @param limit      阀值:  offset为正数时增加后的LongValue值不能大于limit， offset为负数时减少后的value值不能小于limit， 传NULL代表不限制
     * @param expireTime 失效时间, 传 NULL 为默认失效时间 January 1, 2099, 00:00:00 GMT
     * @return 成功/失败
     */
    boolean increase(String key, long offset, Long limit, ExpireTime expireTime);

    /**
     * 删除键对应的记录
     *
     * @param key 键
     * @return 如果键已存在且有效则返回 {@code true};否则返回 {@code false}
     */
    boolean delete(String key);

    /**
     * 批量删除键对应的记录
     *
     * @param keys 需要删除的键组成的list
     * @return 成功删除的有效记录数
     */
    int delete(List<String> keys);

    /**
     * 获取键对应的失效时间
     *
     * @param key 键
     * @return 如果键存在则返回其失效时间，如果不存在返回{@code null}
     */
    ExpireTime getExpire(String key);

    /**
     * 设置键对应的失效时间
     *
     * @param key        键
     * @param expireTime 失效时间
     * @return 如果键存在，返回 {@code true};否则返回 {@code false}
     */
    boolean expire(String key, ExpireTime expireTime);
}

