package cn.com.duiba.kjy.livecenter.api.remoteservice.live;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.kjy.livecenter.api.dto.LiveInfoDto;
import cn.com.duiba.kjy.livecenter.api.dto.aliyunlive.StreamUrlDto;
import cn.com.duiba.kjy.livecenter.api.dto.live.LiveConfRewardPkDto;
import cn.com.duiba.kjy.livecenter.api.dto.live.LiveDetailDto;
import cn.com.duiba.kjy.livecenter.api.dto.live.LiveDto;
import cn.com.duiba.kjy.livecenter.api.dto.live.LiveSimpleInfoDto;
import cn.com.duiba.kjy.livecenter.api.dto.live.LiveTitleAndTimeDto;
import cn.com.duiba.kjy.livecenter.api.dto.live.LiveTitleDto;
import cn.com.duiba.kjy.livecenter.api.dto.live.OuterLiveDto;
import cn.com.duiba.kjy.livecenter.api.param.CreateLiveMpQrCodeParam;
import cn.com.duiba.kjy.livecenter.api.param.live.LiveSearchParam;

import java.util.List;
import java.util.Map;

/**
 * 直播
 * @author lizhi
 * @date 2020/3/30 2:57 PM
 */
@AdvancedFeignClient
public interface RemoteLiveService {

    /**
     *  阿里云回调专用
     *  根据阿里云流名称，获取直播
     * @param streamName 阿里云流名称
     * @return
     */
    LiveDto getLiveByAliyunStream(String streamName);

    /**
     * 根据条件查询，分页
     * @param param 查询条件
     * @return 直播集合
     */
    List<LiveDto> selectPage(LiveSearchParam param);

    /**
     * 根据条件查询，分页
     * @param param 查询条件
     * @return 直播主键集合
     */
    List<Long> selectIdPage(LiveSearchParam param);

    /**
     * 统计
     * @param param
     * @return
     */
    Integer selectCount(LiveSearchParam param);

    /**
     * 从缓存中查询直播详情，包含配置信息
     * @param liveId 直播ID
     * @return 直播详情，包含配置信息
     */
    LiveDetailDto findLiveDetailFromCache(Long liveId);

    /**
     * 新增或更新直播
     * @param dto 直播信息
     * @return 直播ID
     */
    Long saveOrUpdateLive(LiveDto dto);

    /**
     * 修改直播互动红包限额
     * @param liveId 直播ID
     * @param amount 互动红包限额
     * @return 结果
     */
    boolean saveOrUpdateInteractionAmount(Long liveId, Integer amount);

    /**
     * 根据主键查询
     * @param id 主键
     * @return 直播
     */
    LiveDto findById(Long id);

    /**
     * 更新直播状态
     * @param liveId 直播ID
     * @param liveStatus 直播状态
     * @return 更新结果
     */
    int updateStatus(Long liveId, Integer liveStatus);

    /**
     * 新增或更新pk奖励
     * @param liveId 直播ID
     * @param pkDtoList pk奖励集合
     * @return 执行结果
     */
    int batchSaveOrUpdatePk(Long liveId, List<LiveConfRewardPkDto> pkDtoList);

    /**
     * 根据直播ID和类型查询
     * @param liveId 直播ID，不可为空
     * @param rewardType 奖励类型，1-个人，2-团队，可为空
     * @return 直播配置-pk奖励集合
     */
    List<LiveConfRewardPkDto> findByLiveIdAndType(Long liveId, Integer rewardType);

    /**
     * 从缓存获取直播的推流信息
     * @param liveId 直播主键
     * @return 推流信息
     */
    StreamUrlDto getLivePushInfoFromCache(Long liveId);

    /**
     * 从缓存获取直播的拉流信息
     * @param liveId 直播主键
     * @return 拉流信息
     */
    List<StreamUrlDto> getLivePullInfoFromCache(Long liveId);

    /**
     * 根据直播ID查询公司ID
     * @param id
     * @return
     */
    Long findCompanyById(Long id);

    /**
     * 获取生成小程序码的参数（管理员直播列表分享）
     * @param liveId 直播ID
     * @return 生成小程序码的参数
     */
    CreateLiveMpQrCodeParam getMpQrCodeParam(Long liveId);

    /**
     * 更新直播间二维码
     * @param liveId
     * @param qrCodeUrl
     * @return
     */
    Integer updateQrcodeUrl(Long liveId,String qrCodeUrl);

    /**
     * 直播间禁播
     * @param liveId
     * @return
     */
    Boolean forbidLiveStream(Long liveId);

    /**
     * 根据直播id集合获取直播标题
     * @param ids id集合
     * @return key：直播id value：直播标题
     */
    Map<Long,String> findTitleByIds(List<Long> ids);

    /**
     * 根据直播id集合获取直播标题、时间信息
     * @param ids 直播id集合
     * @return key：直播id value：直播标题、时间信息
     */
    Map<Long, LiveTitleAndTimeDto> findTitleAndTimeByIds(List<Long> ids);

    /**
     * 根据直播id集合获取直播信息
     * @param ids 直播id集合
     * @return key：直播id value：直播信息
     */
    Map<Long, LiveSimpleInfoDto> findSimpleInfoByIds(List<Long> ids);

    /**
     * 查询直播的状态。带有本地缓存，跟新时也会自动刷新
     * @param liveId
     * @return
     */
    Integer getLiveStatusWithLocalCache(Long liveId);

    /**
     * 根据直播状态查询某公司下的所有直播
     * @param companyId 公司ID
     * @param liveStatus 状态集合
     * @return
     */
    List<Long> selectByCompanyId(Long companyId, List<Integer> liveStatus);

    /**
     * 根据直播id查询公司id
     * @param liveIds 直播id集合
     * @return key：liveId value：公司id
     */
    Map<Long, Long> findCompanyIdByLiveId(List<Long> liveIds);

    /**
     * 根据公司名获取公司信息，最多返回100条
     * @param title 直播标题
     * @return 公司名称信息
     */
    List<LiveTitleDto> findByTitle(String title);

    /**
     * 根据liveId 查找公司Id
     * @param liveId
     * @return
     */
    Long findCompanyIdByOneLiveId(Long liveId);

    /**
     * 根据用户Id 查询直播间列表
     * 查代理人所在公司的直播列表
     * @param liveUserId 直播系统用户id
     * @return
     */
    List<OuterLiveDto> findListListByLiveUserId4Kjj(Long liveUserId, Integer pageIndex, Integer pageSize);

    /**
     * 查询未结束的最近一场直播的直播信息
     * @param companyId 公司id
     * @return 直播信息
     */
    LiveInfoDto findNotOverLiveIdsByCompanyId(Long companyId);


    /**
     * 根据公司列表和状态列表查询第一场直播
     * 根据开播时间排序
     * @param companyIds
     * @param status
     * @return
     */
    LiveDto selectFirstLiveByCompanyIdsAndStatus(List<Long> companyIds,List<Integer> status);

    /**
     * 删除直播间
     * @param id
     * @return
     */
    Integer deleteById(Long id);
}
