package cn.com.duiba.live.mall.api.response;

import cn.com.duiba.live.mall.api.exception.ErrorCode;
import org.apache.commons.lang3.StringUtils;

/**
 * 构建请求返回数据对象工具类
 * <p>
 * Created by sunchangji on 2018/9/13.
 */
public final class ResultBuilders {
    /**
     * 默认成功码
     */
    private static final String DEFAULT_SUCCESS_CODE = "000000";
    /**
     * 默认错误码
     */
    private static final String DEFAULT_FAIL_CODE = "999999";

    private ResultBuilders() {
        // X Y
    }

    /**
     * 带自定义响应码的描述信息数据对象Result
     *
     * @param code 错误码
     * @param desc 失败描述
     * @return Result
     */
    public static <T> Result<T> fail(String code, String desc) {
        Result<T> result = new Result<>();
        result.setTimestamp(System.currentTimeMillis());
        result.setSuccess(false);
        result.setCode(StringUtils.isBlank(code) ? DEFAULT_FAIL_CODE : code);
        result.setDesc(desc);
        return result;
    }

    /**
     * 使用默认错误码,描述错误信息(统一使用默认错误码)
     *
     * @param desc 失败描述
     * @return Result
     */
    public static <T> Result<T> fail(String desc) {
        return fail(DEFAULT_FAIL_CODE, desc);
    }

    /**
     * 业务处理失败
     * @param code 错误码
     * @param desc 错误描述
     */
    public static <T> Result<T> bizFail(String code,String desc) {
        return fail(code, desc);
    }
    /**
     * 使用默认错误码,描述错误信息(统一使用默认错误码)
     *
     * @param errCode
     * @return Result
     */
    public static <T> Result<T> fail(ErrorCode errCode) {
        return fail(errCode.getCode().toString(), errCode.getDesc());
    }

    /**
     * 使用错误码,描述错误信息
     *
     * @param errCode
     * @return Result
     */
    public static <T> Result<T> fail(ErrorCode errCode, T data) {
        Result<T> result = new Result<>();
        result.setTimestamp(System.currentTimeMillis());
        result.setSuccess(false);
        result.setCode(errCode.getCode().toString());
        result.setDesc(errCode.getDesc());
        result.setData(data);
        return result;
    }

    /**
     * 带有数据的失败返回,统一使用默认错误码
     *
     * @param desc 失败信息
     * @param data 数据
     * @return Result
     */
    public static <T> Result<T> fail(String desc, T data) {
        return fail(DEFAULT_FAIL_CODE, desc, data);
    }

    /**
     * 带有数据的失败返回
     *
     * @param code 失败码
     * @param desc 失败信息
     * @param data 数据
     * @return Result
     */
    public static <T> Result<T> fail(String code, String desc, T data) {
        Result<T> result = new Result<>();
        result.setTimestamp(System.currentTimeMillis());
        result.setSuccess(false);
        result.setCode(code);
        result.setDesc(desc);
        result.setData(data);
        return result;
    }

    /**
     * 使用自定义成功码的成功数据对象构建
     *
     * @param code 成功码
     * @param data 数据
     * @return Result
     */
    public static <T> Result<T> success(String code, T data) {
        Result<T> result = new Result<>();
        result.setTimestamp(System.currentTimeMillis());
        result.setSuccess(true);
        result.setCode(code);
        result.setData(data);
        result.setDesc("OK");
        return result;
    }


    /**
     * 统一使用默认成功码的成功数据对象构建
     *
     * @param data 数据
     * @return Result
     */
    public static <T> Result<T> success(T data) {
        return success(DEFAULT_SUCCESS_CODE, data);
    }


    /**
     * 成功,不返回任何数据,统一使用默认成功码
     *
     * @return
     */
    public static <T> Result<T> success() {
        return success(DEFAULT_SUCCESS_CODE, null);
    }
}
