package cn.com.duiba.log.api.service;

import cn.com.duiba.log.api.appender.LogAppenderService;
import cn.com.duiba.log.api.context.LoggerStackContext;
import cn.com.duiba.log.api.context.LoggerThreadContext;
import com.alibaba.fastjson.JSONObject;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;

/**
 * @Description 日志线程包装类
 * @Author dangcang
 * @Date 2021/11/4 10:19 上午
 **/
@Component
public class LoggerService implements InitializingBean {

    @Resource(name= "logExecutorService")
    private ExecutorService logExecutorService;

    @Autowired
    private List<LogAppenderService> logAppenderServices;

    @Autowired(required = false)
    private LoggerMateInfoAdapter loggerMateInfoAdapter;

    /**
     * 由切面产生,如果没有此日志线程对象,则MyLogger的无法打印日志
     */
    private final ThreadLocal<LoggerStackContext> threadContext = new ThreadLocal<>();

    @Override
    public void afterPropertiesSet() throws Exception {
        MyLogger.setLoggerService(this);
    }

    /**
     * 初始化 stackContext
     *
     * @return
     */
    public LoggerStackContext initStackContext() {
        LoggerStackContext context = threadContext.get();
        if (Objects.isNull(context)) {
            context = new LoggerStackContext();
            threadContext.set(context);
        }
        return context;
    }

    /**
     * 设置 stackContext
     *
     * @param context
     */
    public void setThreadContext(LoggerThreadContext context) {
        LoggerStackContext stackContext = initStackContext();
        if (Objects.isNull(stackContext)) {
            return;
        }
        stackContext.addContext(context);
    }

    /**
     * 通过json渲染日志
     *
     * @param json
     */
    public void log(JSONObject json) {
        Objects.requireNonNull(json);
        LoggerStackContext context = threadContext.get();
        if (Objects.isNull(context)) {
            return;
        }
        context.addLogParam(json);
    }

    /**
     * 打印业务日志
     * 封装 task 异步线程池执行
     */
    public void flushThreadContext() {
        LoggerStackContext stackContext = threadContext.get();
        if (Objects.isNull(stackContext)) {
            return;
        }
        LoggerThreadContext context = stackContext.popContext();

        LoggerRunnable runnable = new LoggerRunnable();
        runnable.setThreadContext(context);
        runnable.setAppenderServiceList(logAppenderServices);
        runnable.setMateInfo(loggerMateInfoAdapter.getLoggerMateInfo());

        logExecutorService.submit(runnable);

        if (stackContext.isEmpty()) {
            threadContext.remove();
        }
    }

}