package cn.com.duiba.duixintong.center.api.enums.statistics;

import lombok.AllArgsConstructor;
import lombok.Getter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.lang3.tuple.Triple;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author pengyi
 * @date 2023/5/19 11:34
 */
@Getter
@AllArgsConstructor
public enum DimensionStatisticsEnum {
    /**
     * 商家+推荐人+活动维度 => businessId_referenceId_activityId
     */
    BUSINESS(1, "商家维度"),
    BUSINESS_REFERENCE(2, "商家+推荐人维度"),
    BUSINESS_ACTIVITY(3, "商家+活动维度"),
    REFERENCE_BUSINESS_ACTIVITY(4, "商家+推荐人+活动维度"),
    ;

    private final Integer dimension;

    private final String desc;
    
    public String buildCombinationId(Long businessId, Long activityId, Long referenceUserId) {
        switch (this) {
            case BUSINESS:
                return doBuildCombinationId(businessId);
            case BUSINESS_REFERENCE:
                return doBuildCombinationId(businessId, referenceUserId);
            case BUSINESS_ACTIVITY:
                return doBuildCombinationId(businessId, activityId);
            case REFERENCE_BUSINESS_ACTIVITY:
                return doBuildCombinationId(businessId, referenceUserId, activityId);
            default:
                return null;
        }
    }

    /**
     * 解析组合ID
     * @param combinationId 组合ID
     * @return left: businessId, middle: activityId, right: referenceUserId
     */
    public Triple<Long, Long, Long> parseCombinationId(String combinationId) {
        if (StringUtils.isBlank(combinationId)) {
            return Triple.of(null, null, null);
        }
        String[] split = combinationId.split(",");
        switch (this) {
            case BUSINESS:
                return Triple.of(NumberUtils.toLong(split[0]), null, null);
            case BUSINESS_REFERENCE:
                return Triple.of(NumberUtils.toLong(split[0]), null, NumberUtils.toLong(split[1]));
            case BUSINESS_ACTIVITY:
                return Triple.of(NumberUtils.toLong(split[0]), NumberUtils.toLong(split[1]), null);
            case REFERENCE_BUSINESS_ACTIVITY:
                return Triple.of(NumberUtils.toLong(split[0]), NumberUtils.toLong(split[2]), NumberUtils.toLong(split[1]));
            default:
                return Triple.of(null, null, null);
        }
    }

    private String doBuildCombinationId(Long... ids) {
        StringBuilder sb = new StringBuilder();
        for (Long id : ids) {
            sb.append(",");
            sb.append(id);
        }
        return sb.substring(1);
    }
    
    public static final Set<DimensionStatisticsEnum> ORDER_STAT_DIMENSION_SET = Collections.unmodifiableSet(Arrays.stream(values()).collect(Collectors.toSet()));

    public static final Set<DimensionStatisticsEnum> CUSTOMER_STAT_DIMENSION_SET = Collections.unmodifiableSet(Stream.of(BUSINESS, BUSINESS_REFERENCE).collect(Collectors.toSet()));
    
    public static final Set<DimensionStatisticsEnum> EQUITY_STAT_DIMENSION_SET = Collections.unmodifiableSet(Stream.of(BUSINESS, BUSINESS_REFERENCE).collect(Collectors.toSet()));
    
    public static final Set<DimensionStatisticsEnum> REFERENCE_STAT_DIMENSION_SET = Collections.unmodifiableSet(Stream.of(BUSINESS).collect(Collectors.toSet()));
    
    private static final Map<Integer, DimensionStatisticsEnum> ENUM_MAP = Collections.unmodifiableMap(Arrays.stream(values()).collect(Collectors.toMap(DimensionStatisticsEnum::getDimension, Function.identity(), (v1, v2) -> v2)));
    
    /**
     * 根据枚举值获取枚举
     * @param dimension 枚举值
     * @return 枚举
     */
    public static DimensionStatisticsEnum getByDimension(Integer dimension) {
        if (dimension == null) {
            return null;
        }
        return ENUM_MAP.get(dimension);
    }
}
