package cn.com.duiba.goods.center.api.remoteservice.item;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.boot.netflix.feign.hystrix.FeignHystrixCommand;
import cn.com.duiba.boot.netflix.feign.hystrix.FeignHystrixProperty;
import cn.com.duiba.boot.netflix.feign.hystrix.conf.HystrixPropertiesManager;
import cn.com.duiba.goods.center.api.remoteservice.dto.AppItemPageDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.CouponModifySkuValidDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.CouponValidDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.AppItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyStockDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemQueryBaseDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemStockQuantityDto;
import cn.com.duiba.goods.center.api.remoteservice.param.AppItemQueryParam;
import cn.com.duiba.goods.center.api.remoteservice.param.StockBizCheckParam;
import cn.com.duiba.wolf.dubbo.DubboResult;
import org.springframework.web.bind.annotation.RequestMapping;

import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 商品封装服务
 * 
 * @author houwen
 */
@AdvancedFeignClient
public interface RemoteItemKeyService {

	/**
	 * 查询商品封装信息，推存用findItemKeyNew
	 * 
	 * @param appItemId 开发者商品ID
	 * @param itemId 兑吧商品ID
	 * @param appId 应用ID
	 * @return
	 *
	 * 自有商品删除时  findItem appitem null
	 */
	@Deprecated
	public DubboResult<ItemKeyDto> findItemKey(Long appItemId, Long itemId, Long appId);

	/**
	 * 严重注意：牵涉到库存等查询实时信息的不要用这个接口，
	 * 需实时商品信息请用findItemKey(Long appItemId, Long itemId, Long appId);
	 *
	 * 通过本地缓存查询商品封装信息
	 * @param appItemId
	 * @param itemId
	 * @param appId
	 * @return
	 */
	DubboResult<ItemKeyDto> findItemKeyByCache(Long appItemId, Long itemId, Long appId);

	/**
	 * 查询商品封装信息,包含已删除的
	 *
	 * @param appItemId 开发者商品ID
	 * @param itemId 兑吧商品ID
	 * @param appId 应用ID
	 * @return
	 */
	public DubboResult<ItemKeyDto> findItemKeyIncludeDeleted(Long appItemId, Long itemId, Long appId);

	/**
	 * 批量查询
	 * @param appItemIds
	 * @param appId
	 * @return
	 */
	public DubboResult<List<ItemKeyDto>> findItemKeyByAppItemIds(List<Long> appItemIds, Long appId);

    /**
     * 批量查询
     * @param itemIds
     * @param appId
     * @return
     */
    public DubboResult<List<ItemKeyDto>> findItemKeyByItemIdsAndAppId(List<Long> itemIds, Long appId);
	/**
	 * 查询商品当前当前库存
	 * 1.商品总库存
	 * 2.APP定向库存
	 * 3.预分配库存
	 * 4.预分配共享库存
	 * 5.每日限量库存
	 * 返回库存最小值
	 * @param itemKeyDto 商品对象
	 * @return
	 */
	public DubboResult<Long> findStock(ItemKeyDto itemKeyDto);

	/**
	 * 批量查询兑吧商品库存
	 * 1.商品总库存
	 * 2.定向库存
	 * 3.预分配库存
	 * 取最小值
	 * @param appItemIds
	 * @return
	 */
	Map<Long, Integer> findStockByIds(Long appId, List<Long> appItemIds) throws BizException;
	
	/**
	 * 批量查询商品当前当前库存
	 * 1.商品总库存
	 * 2.APP定向库存
	 * 3.预分配库存
	 * 4.预分配共享库存
	 * 5.每日限量库存
	 * 返回库存最小值
	 * @param itemKeyDtos 商品对象
	 * @param appId identify a app
	 * @return
	 */
	public DubboResult<List<ItemKeyStockDto>> batchFindStock(List<ItemKeyDto> itemKeyDtos, Long appId);

	/**
	 * 扣一个商品库存
	 * 1.扣商品总库存
	 * 2.扣APP定向库存
	 * 3.扣预分配库存
	 * 4.扣预分配共享库存
	 * 5.扣预分配每日限量库存
	 * 6.扣每日限量库存
	 * @param itemKeyDto
	 * @param bizId
	 * @param bizSource
	 * @return
	 */
	@FeignHystrixCommand(threadPoolKey = "goods-consumestock-pool")
	@FeignHystrixProperty(name= HystrixPropertiesManager.CORE_SIZE, value="80")
	public DubboResult<Boolean> consumeStock(ItemKeyDto itemKeyDto, String bizId, String bizSource);


	/**
	 * 批量扣商品库存，減少 循环调用
	 * 1.扣商品总库存
	 * 2.扣APP定向库存
	 * 3.扣预分配库存
	 * 4.扣预分配共享库存
	 * 5.扣预分配每日限量库存
	 * 6.扣每日限量库存
	 * @param itemKeyDtos
	 * @param bizId
	 * @param bizSource
	 * @return
	 */
	public DubboResult<Boolean> consumeBatchStock(List<ItemKeyDto> itemKeyDtos, String bizId, String bizSource, Date bizDate);

	/**
	 * 批量扣商品库存(指定商品数量)，減少 循环调用
	 * 1.扣商品总库存
	 * 2.扣APP定向库存
	 * 3.扣预分配库存
	 * 4.扣预分配共享库存
	 * 5.扣预分配每日限量库存
	 * 6.扣每日限量库存
	 * @param itemKeyDtos
	 * @param bizId
	 * @param bizSource
	 * @return
	 */
	 DubboResult<Boolean> consumeBatchStockWithNum(List<ItemStockQuantityDto> itemKeyDtos, String bizId, String bizSource, Date bizDate);



	DubboResult<Boolean> consumeBatchStockWithNumAndUid(List<ItemStockQuantityDto> itemKeyDtos, String bizId, String bizSource, Date bizDate, StockBizCheckParam stockBizCheckParam);

	/**
	 * @deprecated
	 * 返还商品库存
	 * 1.返商品总库存
	 * 2.返APP定向库存
	 * 3.返预分配库存
	 * 4.返预分配共享库存
	 * 5.返预分配每日限量库存
	 * 6.返每日限量库存
	 * @param itemKeyDto 商品对象
	 * @param bizId 业务流水号
	 * @param bizSource 业务类型
	 * @return
	 */
	@Deprecated
	public DubboResult<Boolean> rollbackStock(ItemKeyDto itemKeyDto, String bizId, String bizSource);


	/**
	 * @deprecated
	 *   批量  返还商品库存（单个数量）
	 * 1.返商品总库存
	 * 2.返APP定向库存
	 * 3.返预分配库存
	 * 4.返预分配共享库存
	 * 5.返预分配每日限量库存
	 * 6.返每日限量库存
	 * @param itemKeyDtos 商品对象
	 * @param bizId 业务流水号
	 * @param bizSource 业务类型
	 * @return
	 */
	 DubboResult<Boolean> batchRollbackStock(List<ItemKeyDto> itemKeyDtos,  String bizId, String bizSource, Date bizDate);

	/**
	 * @deprecated
	 *   批量  返还商品库存(指定商品数量)
	 * 1.返商品总库存
	 * 2.返APP定向库存
	 * 3.返预分配库存
	 * 4.返预分配共享库存
	 * 5.返预分配每日限量库存
	 * 6.返每日限量库存
	 * @param itemKeyDtos 商品对象
	 * @param bizId 业务流水号
	 * @param bizSource 业务类型
	 * @return
	 */
	DubboResult<Boolean> batchRollbackStockNum(List<ItemStockQuantityDto> itemKeyDtos,  String bizId, String bizSource, Date bizDate);


	/**
	 * 返还商品库存
	 * 1.返商品总库存
	 * 2.返APP定向库存
	 * 3.返预分配库存
	 * 4.返预分配共享库存
	 * 5.返预分配每日限量库存
	 * 6.返每日限量库存
	 * @param itemKeyDto 商品对象
	 * @param bizId 业务流水号
	 * @param bizSource 业务类型
	 * @param bizDate 业务订单创建时间
	 * @return
	 */
	@RequestMapping("rollbackStockByBizDate")
	public DubboResult<Boolean> rollbackStock(ItemKeyDto itemKeyDto, String bizId, String bizSource, Date bizDate);


	/**
	 * 获取开发者秒杀商品
	 * @param appId
	 * @return
	 */
	DubboResult<List<ItemKeyDto>> findDevSecKill(Long appId);

	/**
	 * 获取首页商品区列表(缓存)
	 * @param appId
	 * @return
	 */
	DubboResult<List<ItemKeyDto>> findHomeItemKey(Long appId);

    /**
     * **分页**获取首页商品区列表(缓存)，page从1开始
     * 废弃，请还用 findHomeItemKeyItemTypeWithPagination 方法
     * @param appId
     * @param page
     * @param pageSize
     * @return
     */
    @Deprecated
    DubboResult<List<ItemKeyDto>> findHomeItemKeyWithPagination(Long appId, Integer page, Integer pageSize);

    /**
     * **分页**获取首页商品区列表(缓存)，page从1开始
     * @param appId
     * @param itemType
     * @param page
     * @param pageSize
     * @return
     */
    DubboResult<List<ItemKeyDto>> findHomeItemKeyItemTypeWithPagination(Long appId, String itemType, Integer page, Integer pageSize);

    /**
     * 获取首页商品区总条数(缓存)
     * 废弃，请使用 findHomeItemKeyItemTypeCnt 方法
     * @param appId
     * @return
     */
    @Deprecated
    DubboResult<Long> findHomeItemKeyCnt(Long appId);

    /**
     * 获取首页商品区总条数(缓存)
     * @param appId
	 * @param itemType
     * @return
     */
    DubboResult<Long> findHomeItemKeyItemTypeCnt(Long appId, String itemType);

	/**
	 * 根据appId和classifyId获取首页商品区列表
	 * @param appId
	 * @return
	 */
	List<ItemKeyDto> findHomeItemKeyByClassify(Long appId,Long classifyId);

	/**
	 * 根据appId查询指定数量的已上架商品用于首页展示
	 * 条件：状态为开启且未删除、首页展示开关打开
	 * 排序：按权重由大到小倒叙排列
	 * @param appId
	 * @param limitCount
	 * @return
	 */
	DubboResult<List<ItemKeyDto>> findHomeItemKeyByAppIdAndLimitCount(long appId, int limitCount);

	/**
	 * APP维度通用商品查询接口，param->appId必填
	 * 默认查询未删除的商品
	 * 分页查询:pageSize与pageNum必填
	 * 排序：按权重由大到小倒叙排列
	 * @param param
	 * @author xiaoxuda 2017/07/14
	 * @return
	 */
	DubboResult<List<ItemKeyDto>> findItemKey4App(AppItemQueryParam param);


	/**
	 * APP维度通用商品查询接口，param->appId必填
	 * 默认查询未删除的商品
	 * 分页查询:pageSize与pageNum必填
	 * 排序：按权重由大到小倒叙排列
	 * @param param
	 * @author liujunang 2018/06/01
	 * @return AppItemPageDto
	 */
	AppItemPageDto findItemKeyAppPageInfo(AppItemQueryParam param);

	/**
	 * 根据id批量查询商品项，返回值包含商品项对应的itemDto、在分类信息中的权重、优惠券的库存信息
	 * @param appItemIds
	 * @return
	 */
	DubboResult<List<ItemKeyDto>> getBatchItemKeyByAppItemIds(List<Long> appItemIds);

    /**
     * 根据id批量查询商品项
     * @param appItemIds
     * @return
     */
    DubboResult<List<ItemKeyDto>> getBatchItemKeyByAppItemIdsIncludeDeleted(List<Long> appItemIds);

	/**
	 * getBatchItemKeyByAppItem
	 * @param appItems
	 * @param appId
	 * @return
	 */
	public DubboResult<List<ItemKeyDto>> getBatchItemKeyByAppItem(List<AppItemDto> appItems, Long appId);

	/**
	 * 根据id查询商品项，返回值包含商品项对应的itemDto、在分类信息中的权重、优惠券的库存信息
	 * @param appItemId
	 * @return
	 */
	DubboResult<ItemKeyDto> getItemKey(Long appItemId);

	/**
	 * 根据appId与itemId查询商品项，返回值包含商品项对应的itemDto、在分类信息中的权重、优惠券的库存信息
	 * @param appId
	 * @param itemId
	 * @return
	 */
	@RequestMapping("getItemKeyByAppAndItemId")
	DubboResult<ItemKeyDto> getItemKey(Long appId,Long itemId);

	/**
	 * 获取指定appId及classifyId的商品项数据，包含商品在分类中的权重值,商品券库存
	 * @param appId 不能为空
	 * @param classifyId 分类ID,不能为空
	 * @param status 商品状态，可为空
	 * @return
	 */
	DubboResult<List<ItemKeyDto>> findItemKeysWithClassifyPayLoad(Long appId, Long classifyId, String status);

	/**
	 * 查询商品封装信息
	 *
	 * @param appItemId 开发者商品ID
	 * @param itemId 兑吧商品ID
	 * @param appId 应用ID
	 * @return
	 */
	public ItemKeyDto findItemKeyNew(Long appItemId, Long itemId, Long appId);

	/**
	 * 扣一个商品库存
	 * 1.扣商品总库存
	 * 2.扣APP定向库存
	 * 3.扣预分配库存
	 * 4.扣预分配共享库存
	 * 5.扣预分配每日限量库存
	 * 6.扣每日限量库存
	 * @param itemKeyDto
	 * @param bizId
	 * @param bizSource
	 * @return
	 */
	@FeignHystrixCommand(threadPoolKey = "goods-consumestock-pool")
	@FeignHystrixProperty(name= HystrixPropertiesManager.CORE_SIZE, value="80")
	public Boolean consumeStockNew(ItemKeyDto itemKeyDto, String bizId, String bizSource);
    /**
     * 获取商品主图
     * @param itemKey
     * @param skuId
     * @return
     */
    String getItemMainImage(ItemKeyDto itemKey, Long appSkuId, Long skuId);

	/**
	 * 查询商品列表, 此方法慎用，现只有订单查询商品信息时用
	 * 加上Deprecated，不推荐用，有这方面需求可以自己写新方法实现
	 * @param list
	 * @return
	 */
	@Deprecated
    List<ItemKeyDto> findItemKeyListIncludeDelete(List<ItemQueryBaseDto> list);

	/**
	 * 查询商品列表, 不返回sku及库存信息
	 * @param list
	 * @return
	 */
	List<ItemKeyDto> findItemKeyListIncludeDeleteWithoutSkuStock(List<ItemQueryBaseDto> list) throws BizException;

	/**
	 * 填充前置接口优惠券卡密有效期（优惠券取当前在兑换的批次，卡密取距离当前时间最近的sku的有效期）
	 * @param couponList
	 * @return
	 */
	List<CouponValidDto> findFrontValidEndDate(List<ItemKeyDto> couponList);

	/**
	 * 供应商接口用，返回优惠券卡密各sku有效期和最后更新时间
	 * @param appItemDto
	 * @return
	 */
	CouponModifySkuValidDto findSupplyFrontValidEndDate(AppItemDto appItemDto);

	/**
	 * 查询商品列表, 不返回sku及库存信息
	 * @param appItemId
	 * @param itemId
	 * @param appId
	 * @return
	 */
	ItemKeyDto findItemKeyIncludeDeletedWithoutSkuStock(Long appItemId, Long itemId, Long appId) throws BizException;

	/**
	 * 批量查询商品总库存
	 * @param appItemIds
	 * @return
	 */
	Map<Long, Long> batchFindTotalStock(List<Long> appItemIds);
}
