package cn.com.duiba.goods.center.api.remoteservice.item;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.AppItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyStockDto;
import cn.com.duiba.goods.center.api.remoteservice.param.AppItemQueryParam;
import cn.com.duiba.wolf.dubbo.DubboResult;
import org.springframework.web.bind.annotation.RequestMapping;

import java.util.Date;
import java.util.List;

/**
 * 商品封装服务
 * 
 * @author houwen
 */
@AdvancedFeignClient
public interface RemoteItemKeyService {

	/**
	 * 查询商品封装信息
	 * 
	 * @param appItemId 开发者商品ID
	 * @param itemId 兑吧商品ID
	 * @param appId 应用ID
	 * @return
	 */
	public DubboResult<ItemKeyDto> findItemKey(Long appItemId, Long itemId, Long appId);

	/**
	 * 严重注意：牵涉到库存等查询实时信息的不要用这个接口，
	 * 需实时商品信息请用findItemKey(Long appItemId, Long itemId, Long appId);
	 *
	 * 通过本地缓存查询商品封装信息
	 * @param appItemId
	 * @param itemId
	 * @param appId
	 * @return
	 */
	DubboResult<ItemKeyDto> findItemKeyByCache(Long appItemId, Long itemId, Long appId);

	/**
	 * 查询商品封装信息,包含已删除的
	 *
	 * @param appItemId 开发者商品ID
	 * @param itemId 兑吧商品ID
	 * @param appId 应用ID
	 * @return
	 */
	public DubboResult<ItemKeyDto> findItemKeyIncludeDeleted(Long appItemId, Long itemId, Long appId);

	/**
	 * 批量查询
	 * @param appItemIds
	 * @param appId
	 * @return
	 */
	public DubboResult<List<ItemKeyDto>> findItemKeyByAppItemIds(List<Long> appItemIds, Long appId);

	/**
	 * 查询商品当前当前库存
	 * 1.商品总库存
	 * 2.APP定向库存
	 * 3.预分配库存
	 * 4.预分配共享库存
	 * 5.每日限量库存
	 * 返回库存最小值
	 * @param itemKeyDto 商品对象
	 * @return
	 */
	public DubboResult<Long> findStock(ItemKeyDto itemKeyDto);
	
	/**
	 * 批量查询商品当前当前库存
	 * 1.商品总库存
	 * 2.APP定向库存
	 * 3.预分配库存
	 * 4.预分配共享库存
	 * 5.每日限量库存
	 * 返回库存最小值
	 * @param itemKeyDtos 商品对象
	 * @param appId identify a app
	 * @return
	 */
	public DubboResult<List<ItemKeyStockDto>> batchFindStock(List<ItemKeyDto> itemKeyDtos, Long appId);

	/**
	 * 扣一个商品库存
	 * 1.扣商品总库存
	 * 2.扣APP定向库存
	 * 3.扣预分配库存
	 * 4.扣预分配共享库存
	 * 5.扣预分配每日限量库存
	 * 6.扣每日限量库存
	 * @param itemKeyDto
	 * @param bizId
	 * @param bizSource
	 * @return
	 */
	public DubboResult<Boolean> consumeStock(ItemKeyDto itemKeyDto, String bizId, String bizSource);

	/**
	 * @deprecated
	 * 返还商品库存
	 * 1.返商品总库存
	 * 2.返APP定向库存
	 * 3.返预分配库存
	 * 4.返预分配共享库存
	 * 5.返预分配每日限量库存
	 * 6.返每日限量库存
	 * @param itemKeyDto 商品对象
	 * @param bizId 业务流水号
	 * @param bizSource 业务类型
	 * @return
	 */
	@Deprecated
	public DubboResult<Boolean> rollbackStock(ItemKeyDto itemKeyDto, String bizId, String bizSource);
	
	/**
	 * 返还商品库存
	 * 1.返商品总库存
	 * 2.返APP定向库存
	 * 3.返预分配库存
	 * 4.返预分配共享库存
	 * 5.返预分配每日限量库存
	 * 6.返每日限量库存
	 * @param itemKeyDto 商品对象
	 * @param bizId 业务流水号
	 * @param bizSource 业务类型
	 * @param bizDate 业务订单创建时间
	 * @return
	 */
	@RequestMapping("rollbackStockByBizDate")
	public DubboResult<Boolean> rollbackStock(ItemKeyDto itemKeyDto, String bizId, String bizSource, Date bizDate);


	/**
	 * 获取开发者秒杀商品
	 * @param appId
	 * @return
	 */
	DubboResult<List<ItemKeyDto>> findDevSecKill(Long appId);

	/**
	 * 获取首页商品区列表(缓存)
	 * @param appId
	 * @return
	 */
	DubboResult<List<ItemKeyDto>> findHomeItemKey(Long appId);

	/**
	 * 根据appId查询指定数量的已上架商品用于首页展示
	 * 条件：状态为开启且未删除、首页展示开关打开
	 * 排序：按权重由大到小倒叙排列
	 * @param appId
	 * @param limitCount
	 * @return
	 */
	DubboResult<List<ItemKeyDto>> findHomeItemKeyByAppIdAndLimitCount(long appId, int limitCount);

	/**
	 * APP维度通用商品查询接口，param->appId必填
	 * 默认查询未删除的商品
	 * 分页查询:pageSize与pageNum必填
	 * 排序：按权重由大到小倒叙排列
	 * @param param
	 * @author xiaoxuda 2017/07/14
	 * @return
	 */
	DubboResult<List<ItemKeyDto>> findItemKey4App(AppItemQueryParam param);

	/**
	 * 根据id批量查询商品项，返回值包含商品项对应的itemDto、在分类信息中的权重、优惠券的库存信息
	 * @param appItemIds
	 * @return
	 */
	DubboResult<List<ItemKeyDto>> getBatchItemKeyByAppItemIds(List<Long> appItemIds);
	
	/**
	 * getBatchItemKeyByAppItem
	 * @param appItems
	 * @param appId
	 * @return
	 */
	public DubboResult<List<ItemKeyDto>> getBatchItemKeyByAppItem(List<AppItemDto> appItems, Long appId);

	/**
	 * 根据id查询商品项，返回值包含商品项对应的itemDto、在分类信息中的权重、优惠券的库存信息
	 * @param appItemId
	 * @return
	 */
	DubboResult<ItemKeyDto> getItemKey(Long appItemId);

	/**
	 * 根据appId与itemId查询商品项，返回值包含商品项对应的itemDto、在分类信息中的权重、优惠券的库存信息
	 * @param appId
	 * @param itemId
	 * @return
	 */
	@RequestMapping("getItemKeyByAppAndItemId")
	DubboResult<ItemKeyDto> getItemKey(Long appId,Long itemId);

	/**
	 * 获取指定appId及classifyId的商品项数据，包含商品在分类中的权重值,商品券库存
	 * @param appId 不能为空
	 * @param classifyId 分类ID,不能为空
	 * @param status 商品状态，可为空
	 * @return
	 */
	DubboResult<List<ItemKeyDto>> findItemKeysWithClassifyPayLoad(Long appId, Long classifyId, String status);

}
