package cn.com.duiba.goods.center.api.remoteservice.tool.sku;

import cn.com.duiba.goods.center.api.remoteservice.dto.sku.SkuAttributeNode;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * @Desc 抽象sku解析器
 * @Author xuwei
 * @Date 2018/11/12
 */
public abstract class AbstractSkuResolver<T, R extends Serializable> implements SkuResolver<T, R> {
    //规格属性分割符
    private static final String ATTRIBUTE_SEPARATOR = ":";
    //规格属性组分割符
    private static final String ATTRIBUTE_GROUP_SEPARATOR = ";";
    //无规格默认
    private static final String NO_ATTRIBUTE_VALUE = "0:0";
    //默认规格
    private static final Long DEFAULT_ATTRIBUTE_ID = 0L;
    //默认属性
    private static final Long DEFAULT_ATTRIBUTE_VALUE_ID = 0L;

    @Override
    public List<T> resolveNodes(Long appItemId, Long itemId, List<SkuAttributeNode<R>> nodes) {
        return resolveNodes(appItemId, itemId, nodes, null);
    }

    @Override
    public List<T> resolveNodes(Long appItemId, Long itemId, List<SkuAttributeNode<R>> nodes, ResolverListener<T> listener) {
        if (CollectionUtils.isEmpty(nodes)) {
            return Collections.emptyList();
        }
        List<T> skuList = new ArrayList<>();
        for (SkuAttributeNode<R> node : nodes) {
            resolveNode(appItemId, itemId, skuList, StringUtils.EMPTY, node, listener);
        }
        return skuList;
    }

    @Override
    public List<T> resolveNode(Long appItemId, Long itemId, SkuAttributeNode<R> node) {
        if (node == null) {
            return Collections.emptyList();
        }
        List<T> skuList = new ArrayList<>();
        resolveNode(appItemId, itemId, skuList, StringUtils.EMPTY, node, null);
        return skuList;
    }

    /**
     * 解析单节点SKU
     *
     * @param appItemId
     * @param itemId
     * @param skuList
     * @param attributeJson
     * @param node
     * @return
     */
    private void resolveNode(Long appItemId, Long itemId, List<T> skuList, String attributeJson, SkuAttributeNode<R> node, ResolverListener<T> listener) {
        Long attrId = NumberUtils.toLong(node.getSpecId(), DEFAULT_ATTRIBUTE_ID);
        Long valueId = NumberUtils.toLong(node.getId(), DEFAULT_ATTRIBUTE_VALUE_ID);
        //是否为叶子节点
        if (CollectionUtils.isEmpty(node.getChildren())) {
            R skuInfo = getSkuInfo(node);
            //如果销售信息不为null，解析sku
            if (skuInfo != null) {
                attributeJson = StringUtils.join(attributeJson, attrId, ATTRIBUTE_SEPARATOR, valueId);
                T dto = toDto(appItemId, itemId, attributeJson, skuInfo);
                skuList.add(dto);
                if (listener != null) {
                    listener.onCreated(dto);
                }
            }
        } else {
            attributeJson = StringUtils.join(attributeJson, attrId, ATTRIBUTE_SEPARATOR, valueId, ATTRIBUTE_GROUP_SEPARATOR);
            //如果不是叶节点，继续递归
            for (SkuAttributeNode<R> child : node.getChildren()) {
                //图片级联
                child.setImgUrl(node.getImgUrl());
                resolveNode(appItemId, itemId, skuList, attributeJson, child, listener);
            }
        }
    }

    /**
     * 获取skuInfo，包含图片处理
     *
     * @param node
     * @return
     */
    protected R getSkuInfo(SkuAttributeNode<R> node) {
        return node.getSkuInfo();
    }

    /**
     * 创建sku实例
     *
     * @param appItemId
     * @param itemId
     * @param attributeJson
     * @param skuInfo
     * @return
     */
    protected abstract T toDto(Long appItemId, Long itemId, String attributeJson, R skuInfo);

    @Override
    public List<SkuAttributeNode<R>> reverse(List<T> skuList) {
        return reverse(skuList, null);
    }

    @Override
    public List<SkuAttributeNode<R>> reverse(List<T> skuList, NodeListerner<R> listener) {
        if (CollectionUtils.isEmpty(skuList)) {
            return Collections.emptyList();
        }
        List<SkuAttributeNode<R>> nodes = new ArrayList<>();
        for (T sku : skuList) {
            reverseSku(sku, nodes, listener);
        }
        return nodes;
    }

    /**
     * 获取attributeJson字符串
     *
     * @param sku
     * @return
     */
    protected abstract String getAttributeJson(T sku);

    /**
     * 创建skuInfo实例
     *
     * @param sku
     * @return
     */
    protected abstract R toVo(T sku);

    /**
     * 把sku信息解析到对应节点
     *
     * @param sku
     * @param nodes
     */
    private void reverseSku(T sku, List<SkuAttributeNode<R>> nodes, NodeListerner<R> listener) {
        String attributeJson = getAttributeJson(sku);
        if (StringUtils.isBlank(attributeJson)) {
            attributeJson = NO_ATTRIBUTE_VALUE;
        }
        String[] groups = attributeJson.split(ATTRIBUTE_GROUP_SEPARATOR);
        //前节点
        SkuAttributeNode<R> prevNode = null;
        //头节点，用于设置图片
        SkuAttributeNode<R> headNode = null;
        for (int i = 0, last = groups.length - 1; i < groups.length; i++) {
            String[] attrs = groups[i].split(ATTRIBUTE_SEPARATOR);
            String attrId = attrs[0];
            String valueId = attrs[1];
            boolean isLeaf = i >= last;
            if (prevNode == null) {
                headNode = prevNode = getOrCreateNode(attrId, valueId, sku, nodes, isLeaf, listener);
            } else {
                prevNode = getOrCreateNode(attrId, valueId, sku, prevNode.getChildren(), isLeaf, listener);
            }
            postNode(headNode, prevNode);
        }
    }

    /**
     * 主要用于图片链接处理
     *
     * @param headNode
     * @param node
     */
    protected void postNode(SkuAttributeNode<R> headNode, SkuAttributeNode<R> node) {

    }

    /**
     * 获取或者创建节点
     *
     * @param attrId
     * @param valueId
     * @param sku
     * @param nodes
     * @param isLeaf  是否叶子节点
     * @return
     */
    private SkuAttributeNode<R> getOrCreateNode(String attrId, String valueId, T sku, List<SkuAttributeNode<R>> nodes, boolean isLeaf, NodeListerner<R> listener) {
        SkuAttributeNode<R> node = getNode(valueId, nodes);
        if (node == null) {
            node = newNode(attrId, valueId, sku, isLeaf);
            nodes.add(node);
            if (listener != null) {
                listener.onCreated(node);
            }
        }
        return node;
    }

    /**
     * 获取节点
     *
     * @param valueId
     * @param nodes
     * @return
     */
    private SkuAttributeNode<R> getNode(String valueId, List<SkuAttributeNode<R>> nodes) {
        for (SkuAttributeNode<R> node : nodes) {
            if (Objects.equals(valueId, node.getId())) {
                return node;
            }
        }
        return null;
    }

    /**
     * 创建节点
     *
     * @param specId
     * @param valueId
     * @param sku
     * @param isLeaf  是否叶子节点
     * @return
     */
    private SkuAttributeNode<R> newNode(String specId, String valueId, T sku, boolean isLeaf) {
        SkuAttributeNode<R> node = new SkuAttributeNode<>();
        node.setId(valueId);
        node.setSpecId(specId);
        if (!isLeaf) {
            node.setChildren(new ArrayList<>());
        } else {
            R vo = toVo(sku);
            node.setSkuInfo(vo);
        }
        return node;
    }

    protected Long parseString2Long(String value) {
        return StringUtils.isBlank(value) ? null : Long.valueOf(value.trim());
    }

    protected String parseLong2String(Long value) {
        return value == null ? null : value.toString();
    }
}
