/**
 * Project Name:goods-center-api
 * File Name:RemoteGoodsCouponBackendService.java
 * Package Name:cn.com.duiba.goods.center.api.remoteservice
 * Date:2016年5月23日上午10:48:53
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.goods.center.api.remoteservice;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.ACGStockDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsBatchDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsBatchImportLogDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsCouponDto;
import cn.com.duiba.goods.center.api.remoteservice.enums.CouponShowTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.tool.Page;
import cn.com.duiba.wolf.dubbo.DubboResult;
import org.springframework.web.bind.annotation.RequestMapping;

import java.util.Date;
import java.util.List;

/**
 * ClassName:RemoteGoodsCouponBackendService <br/>
 * Function: 券码库，后台调用服务 <br/>
 * Date:     2016年5月23日 上午10:48:53 <br/>
 * @author   xuhengfei
 * @version  
 * @since    JDK 1.6
 * @see 	 
 */
@AdvancedFeignClient
public interface RemoteGoodsCouponBackendService {
    /**
     * 
     * createNormalBatch:创建一个普通的优惠券批次. <br/>
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param start 有效期开始时间
     * @param end  有效期截止时间
     * @return 批次id
     * @since JDK 1.6
     */
    public DubboResult<Long> createNormalBatch(GoodsTypeEnum gtype,long gid,Date start,Date end);
    /**
     * 创建链接类型批次
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param start 开始时间
     * @param end 结束时间
     * @param link 链接内容
     * @param stock 初始化库存
     * @return 批次id
     * @since JDK 1.6
     */
    public DubboResult<Long> createLinkBatch(GoodsTypeEnum gtype,long gid,Date start,Date end,String link,long stock);
    /**
     * 创建重复类型批次
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param start 开始时间
     * @param end 结束时间
     * @param code 卡号
     * @param password 密码
     * @param stock 初始化库存
     * @return 批次id
     * @since JDK 1.6
     */
    public DubboResult<Long> createRepeatBatch(GoodsTypeEnum gtype,long gid,Date start,Date end,String code,String password,long stock);
    /**
     * 更新链接批次的链接和库存
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @param link 链接内容
     * @param changeStock 库存变化值，可以为负数
     * @return 是否成功
     * @since JDK 1.6
     */
    public DubboResult<Boolean> updateLinkBatch(GoodsTypeEnum gtype,long gid,Long batchId,String link,long changeStock);
    /**
     * 更新重复类型批次的卡密和库存
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @param code 卡号
     * @param password 密码
     * @param changeStock 库存变化值，可以为负数
     * @return 是否成功
     * @since JDK 1.　　　
     */
    public DubboResult<Boolean> updateRepeatBatch(GoodsTypeEnum gtype,long gid,Long batchId,String code,String password,long changeStock);


    /**
     * 修改广告商品的有效期（即广告投放时段）
     * @param gtype 商品类型 @see GoodsTypeEnum
     * @param gid 商品ID
     * @param start
     * @param end
     * @return
     */
    public DubboResult<Boolean> updateBatchDay(GoodsTypeEnum gtype, long gid, Date start, Date end);
    /**
     * 查询商品的所有批次
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @return 结果集
     * @since JDK 1.6
     */
    public DubboResult<List<GoodsBatchDto>> findBatchs(GoodsTypeEnum gtype,long gid);
    /**
     * 
     * importNormalCoupons:导入普通券码 <br/>
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @param downloadUrl 券码附件的下载地址
     * @return 导入券码的日志id，通过此id再调用 findBatchImportLog 来查询进度和结果
     * @since JDK 1.6
     */
    public DubboResult<Long> importNormalCoupons(GoodsTypeEnum gtype,long gid,Long batchId,String downloadUrl);
    /**
     * 查询导入的进度个结果
     *
     * @author xuhengfei
     * @param goodsBatchImportLogId 批次导入的日子id
     * @return 批次导入信息
     * @since JDK 1.6
     */
    public DubboResult<GoodsBatchImportLogDto> findBatchImportLog(Long goodsBatchImportLogId);
    /**
     * 删除批次<br/>
     * 正在导入中的批次无法删除
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @return 是否成功
     * @since JDK 1.6
     */
    public DubboResult<Boolean> deleteBatch(GoodsTypeEnum gtype,long gid,Long batchId);//逻辑删除
    /**
     * 
     * 删除批次下未使用的优惠劵(如果删除数量大于5W，则拒绝执行，因为大数据量删除会引发性能问题)
     * @author xuhengfei
     * @param gtype
     * @param gid
     * @param batchId
     * @return
     * @since JDK 1.6
     */
    public DubboResult<Boolean> deleteBatchUnusedCoupons(GoodsTypeEnum gtype,long gid,Long batchId);
    
    /**
     * 查询批次下未使用的优惠券
     * @param gtype
     * @param gid
     * @param batchId
     * @param limit
     * @return
     */
    public DubboResult<List<GoodsCouponDto>> findUnusedCoupons(GoodsTypeEnum gtype,long gid,Long batchId, int limit);
    
    /**
     * 根据ID删除批次下指定数量的未使用的优惠券
     * @param gtype
     * @param gid
     * @param batchId
     * @param couponIds
     * @return
     */
    @RequestMapping("deleteBatchUnusedCouponByCouponIdList")
    public DubboResult<Boolean> deleteBatchUnusedCoupons(GoodsTypeEnum gtype,long gid,Long batchId, List<Long> couponIds);
    
    /**
     * 删除一张未使用的优惠劵
     * @author xuhengfei
     * @param gtype
     * @param gid
     * @param couponId
     * @return
     * @since JDK 1.6
     */
    public DubboResult<Boolean> deleteUnusedCoupon(GoodsTypeEnum gtype,long gid,Long couponId);
    
    /**
     * 分页查询
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @param pageSize 分页大小
     * @param pageIndex 页码
     * @return 结果集
     * @since JDK 1.6
     */
    public DubboResult<Page<GoodsCouponDto>> findPage(GoodsTypeEnum gtype,long gid,long batchId,int pageSize,int pageIndex);
    /**
     * 查询批次的库存
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @return 批次信息
     * @since JDK 1.6
     */
    public DubboResult<GoodsBatchDto> findBatchStock(GoodsTypeEnum gtype,long gid,long batchId);
    /**
     * 根据券码查询 某个批次下的优惠劵
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @param code 卡号
     * @return
     * @since JDK 1.6
     */
    public DubboResult<List<GoodsCouponDto>> findCouponByCode(GoodsTypeEnum gtype,long gid,long batchId,String code);
    
    /**
     * 批量查询商品库存(当前库存和初始库存).
     *
     * @param gtype 商品类型
     * @param acgIds 优惠卷ID列表
     * @return the dubbo result< list< acg stock dto>>
     */
    DubboResult<List<ACGStockDto>> batchFindGoodsStock(GoodsTypeEnum gtype, List<Long> acgIds);
    
    /**
     * 查询商品库存(当前库存和初始库存).
     *
     * @param gtype 商品类型
     * @param acgId 优惠卷ID
     * @return the dubbo result< acg stock dto>
     */
    DubboResult<ACGStockDto> findGoodsStock(GoodsTypeEnum gtype, Long acgId);
    
    /**
     * 更新批次有效期.
     *
     * @param gtype 商品类型
     * @param acgId 商品ID
     * @param batchId 批次ID
     * @param startDay 生效期
     * @param endDay 失效期
     * @return the dubbo result< boolean>
     */
    DubboResult<Boolean> updateValidDate(GoodsTypeEnum gtype, long acgId, long batchId, Date startDay, Date endDay);

    /**
     * 删除某个优惠券指定批次下所有券码.
     *
     * @param gtype 商品类型
     * @param acgId 商品ID
     * @param batchId 批次ID
     * @return the dubbo result< boolean>
     */
    DubboResult<Boolean> deleteGoodsCoupon(GoodsTypeEnum gtype, long acgId, long batchId);
    
    /**
     * 查询单张coupon的信息
     * @author xuhengfei
     * @param gtype
     * @param gid
     * @param couponId
     * @return
     * @since JDK 1.6
     */
    public DubboResult<GoodsCouponDto> findCoupon(GoodsTypeEnum gtype,long gid,long couponId);

    /**
     * 查询链接券或重复券券码
     *
     * @param gtype
     * @param gid
     * @return
     */
    public DubboResult<GoodsCouponDto> findLinkOrRepateCoupon(GoodsTypeEnum gtype, long gid);
    
    /**
     * 因为唯一索引限制，物理删除批次数据
     *
     * @author zufengze
     * @param gtype
     * @param gid
     * @param batchId
     * @return
     * @since JDK 1.6
     */
    DubboResult<Boolean> physicalDeleteBatch(GoodsTypeEnum gtype, long gid, Long batchId);

    /**
     * 更新优惠券批次展示类型
     * @param showType
     * @param batchId
     * @return
     */
    DubboResult<Boolean> updateBatchShowType(CouponShowTypeEnum showType, Long batchId);

    DubboResult<List<GoodsBatchDto>> findNotDeletedBatchsWithoutAnyCache(GoodsTypeEnum gtype, long gid);

    /**
     * 根据券码查询和状态 某个批次下的优惠劵
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @param code 卡号
     * @return
     * @since JDK 1.6
     */
     DubboResult<List<GoodsCouponDto>> findCouponByCodeAndStatus(GoodsTypeEnum gtype,long gid,long batchId,String code,Integer status);

    /**
     * 根据券码查询和状态 某个批次下的优惠劵
     *
     * @param gtype     商品类型
     * @param gid       商品id
     * @param batchId   批次id
     * @param code      卡号
     * @param pageNo
     * @param pageSize
     * @param withTotal
     * @return
     * @author xuhengfei
     * @since JDK 1.6
     */
    Page<GoodsCouponDto> page(GoodsTypeEnum gtype, long gid, Long batchId, String code, Integer status, int pageNo, int pageSize, boolean withTotal);

}

