package cn.com.duiba.goods.center.api.remoteservice.util;

import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.goods.center.api.remoteservice.enums.VersionPlanEnum;
import org.apache.commons.lang3.StringUtils;

/**
 * 旺店通URL工具类
 * 根据版本套餐自动判断环境并获取对应的URL地址
 * 
 * @author cgq
 * @date 2024-12-19
 */
public class WdtUrlUtil {

    /**
     * 企业版订单推送URL - 测试环境
     */
    private static final String ENTERPRISE_PUSH_URL_TEST = "https://sandbox.wangdian.cn/openapi2/trade_push.php";

    /**
     * 企业版订单推送URL - 正式环境
     */
    private static final String ENTERPRISE_PUSH_URL_PROD = "https://api.wangdian.cn/openapi2/trade_push.php";

    /**
     * 企业版物流查询URL - 测试环境
     */
    private static final String ENTERPRISE_LOGISTICS_QUERY_URL_TEST = "https://sandbox.wangdian.cn/openapi2/logistics_sync_query.php";

    /**
     * 企业版物流查询URL - 正式环境
     */
    private static final String ENTERPRISE_LOGISTICS_QUERY_URL_PROD = "https://api.wangdian.cn/openapi2/logistics_sync_query.php";

    /**
     * 旗舰版基础URL - 测试环境
     */
    private static final String FLAGSHIP_BASE_URL_TEST = "http://47.92.239.46/openapi";

    /**
     * 旗舰版基础URL - 正式环境
     */
    private static final String FLAGSHIP_BASE_URL_PROD = "http://wdt.wangdian.cn/openapi";

    /**
     * 旗舰版订单推送接口
     */
    private static final String FLAGSHIP_PUSH_INTERFACE = "sales.RawTrade.pushSelf";

    /**
     * 旗舰版物流同步信息查询接口
     */
    private static final String FLAGSHIP_LOGISTICS_QUERY_INTERFACE = "sales.LogisticsSync.getSyncListExt";

    /**
     * 判断是否为测试环境
     * 
     * @return true-测试环境，false-正式环境
     */
    private static boolean isTestEnv() {
        return SpringEnvironmentUtils.isTestEnv();
    }

    /**
     * 根据版本套餐获取订单推送URL
     * 自动根据当前环境判断使用测试环境或正式环境URL
     * 
     * @param versionPlan 版本套餐枚举
     * @return 订单推送URL
     */
    public static String getPushUrl(VersionPlanEnum versionPlan) {
        if (versionPlan == null) {
            return null;
        }
        boolean isTest = isTestEnv();
        switch (versionPlan) {
            case ENTERPRISE:
                return isTest ? ENTERPRISE_PUSH_URL_TEST : ENTERPRISE_PUSH_URL_PROD;
            case FLAGSHIP:
                String baseUrl = isTest ? FLAGSHIP_BASE_URL_TEST : FLAGSHIP_BASE_URL_PROD;
                return baseUrl + "/" + FLAGSHIP_PUSH_INTERFACE;
            default:
                return null;
        }
    }

    /**
     * 根据版本套餐值获取订单推送URL
     * 自动根据当前环境判断使用测试环境或正式环境URL
     * 
     * @param versionPlanValue 版本套餐值：1-企业版，2-旗舰版
     * @return 订单推送URL
     */
    public static String getPushUrl(Integer versionPlanValue) {
        if (versionPlanValue == null) {
            return null;
        }
        VersionPlanEnum versionPlan = VersionPlanEnum.fromValue(versionPlanValue);
        return getPushUrl(versionPlan);
    }

    /**
     * 根据版本套餐获取物流查询URL
     * 自动根据当前环境判断使用测试环境或正式环境URL
     * 
     * @param versionPlan 版本套餐枚举
     * @return 物流查询URL
     */
    public static String getLogisticsQueryUrl(VersionPlanEnum versionPlan) {
        if (versionPlan == null) {
            return null;
        }
        boolean isTest = isTestEnv();
        switch (versionPlan) {
            case ENTERPRISE:
                return isTest ? ENTERPRISE_LOGISTICS_QUERY_URL_TEST : ENTERPRISE_LOGISTICS_QUERY_URL_PROD;
            case FLAGSHIP:
                String baseUrl = isTest ? FLAGSHIP_BASE_URL_TEST : FLAGSHIP_BASE_URL_PROD;
                return baseUrl + "/" + FLAGSHIP_LOGISTICS_QUERY_INTERFACE;
            default:
                return null;
        }
    }

    /**
     * 根据版本套餐值获取物流查询URL
     * 自动根据当前环境判断使用测试环境或正式环境URL
     * 
     * @param versionPlanValue 版本套餐值：1-企业版，2-旗舰版
     * @return 物流查询URL
     */
    public static String getLogisticsQueryUrl(Integer versionPlanValue) {
        if (versionPlanValue == null) {
            return null;
        }
        VersionPlanEnum versionPlan = VersionPlanEnum.fromValue(versionPlanValue);
        return getLogisticsQueryUrl(versionPlan);
    }

    /**
     * 根据URL类型和版本套餐获取URL
     * 自动根据当前环境判断使用测试环境或正式环境URL
     * 用于扩展其他类型的URL
     * 
     * @param urlType URL类型：push-订单推送URL，logistics-物流查询URL
     * @param versionPlan 版本套餐枚举
     * @return 对应的URL
     */
    public static String getUrl(String urlType, VersionPlanEnum versionPlan) {
        if (StringUtils.isBlank(urlType) || versionPlan == null) {
            return null;
        }
        switch (urlType.toLowerCase()) {
            case "push":
                return getPushUrl(versionPlan);
            case "logistics":
            case "logistics_query":
                return getLogisticsQueryUrl(versionPlan);
            default:
                return null;
        }
    }

    /**
     * 根据URL类型和版本套餐值获取URL
     * 自动根据当前环境判断使用测试环境或正式环境URL
     * 用于扩展其他类型的URL
     * 
     * @param urlType URL类型：push-订单推送URL，logistics-物流查询URL
     * @param versionPlanValue 版本套餐值：1-企业版，2-旗舰版
     * @return 对应的URL
     */
    public static String getUrl(String urlType, Integer versionPlanValue) {
        if (StringUtils.isBlank(urlType) || versionPlanValue == null) {
            return null;
        }
        VersionPlanEnum versionPlan = VersionPlanEnum.fromValue(versionPlanValue);
        return getUrl(urlType, versionPlan);
    }

    /**
     * 获取旗舰版基础URL
     * 自动根据当前环境判断使用测试环境或正式环境URL
     * 
     * @return 旗舰版基础URL
     */
    public static String getFlagshipBaseUrl() {
        boolean isTest = isTestEnv();
        return isTest ? FLAGSHIP_BASE_URL_TEST : FLAGSHIP_BASE_URL_PROD;
    }

    /**
     * 获取旗舰版订单推送接口名
     * 
     * @return 订单推送接口名
     */
    public static String getFlagshipPushInterface() {
        return FLAGSHIP_PUSH_INTERFACE;
    }

    /**
     * 获取旗舰版物流同步信息查询接口名
     * 
     * @return 物流同步信息查询接口名
     */
    public static String getFlagshipLogisticsQueryInterface() {
        return FLAGSHIP_LOGISTICS_QUERY_INTERFACE;
    }
}
