/**
 * Project Name:goods-center-biz
 * File Name:GoodsBatchCouponBackendBOImpl.java
 * Package Name:cn.com.duiba.goods.center.biz.bo.impl
 * Date:2016年6月5日下午1:25:44
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.goods.center.biz.bo.impl;

import java.io.File;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;

import net.sf.cglib.beans.BeanCopier;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsBatchDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsBatchImportLogDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsCouponDto;
import cn.com.duiba.goods.center.api.remoteservice.tool.Page;
import cn.com.duiba.goods.center.biz.bo.GoodsBatchCouponBackendBO;
import cn.com.duiba.goods.center.biz.dao.GoodsCouponDao.CouponFormat;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchImportLogEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchImportLogService;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.biz.service.PlatformCouponGoodsService;
import cn.com.duiba.goods.center.biz.util.IdSuffixTool;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.GoodsException;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.idmaker.service.api.remoteservice.RemoteIDMakerBackendService;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.wolf.dubbo.DubboResult;

import com.google.common.io.Files;

/**
 * ClassName:GoodsBatchCouponBackendBOImpl <br/>
 * Date:     2016年6月5日 下午1:25:44 <br/>
 * @author   xuhengfei
 * @version  
 * @since    JDK 1.6
 * @see 	 
 */
@Service("goodsBatchCouponBackendBO")
public class GoodsBatchCouponBackendBOImpl implements GoodsBatchCouponBackendBO{
    
    private static final Logger log=LoggerFactory.getLogger(GoodsBatchCouponBackendBOImpl.class);
    @Autowired
    protected GoodsCouponService goodsCouponService;
    @Autowired
    protected GoodsBatchService goodsBatchService;
    @Autowired
    protected PlatformCouponGoodsService platformCouponGoodsService;
    @Autowired
    private RemoteStockBackendService remoteStockBackendService;
    @Autowired
    private GoodsBatchImportLogService goodsBatchImportLogService;
    @Autowired
    private RemoteIDMakerBackendService remoteIDMakerBackendService;
    @Override
    public Long createNormalBatch(GoodsTypeEnum gtype, long gid, Date start, Date end) throws GoodsException{
        DubboResult<Long> stockRet=remoteStockBackendService.newStock(gid, 0);
        if(!stockRet.isSuccess()){
            throw new GoodsException(ErrorCode.E0203003.getErrorCode(), stockRet.getMsg());
        }
        
        Long batchId=goodsBatchService.createNormalBatch(gtype, gid, start, end,stockRet.getResult());
        platformCouponGoodsService.editPlatformCouponType(gid, GoodsBatchEntity.BatchTypeNormal);
        return batchId;
    }

    @SuppressWarnings("deprecation")
    @Override
    public Long createLinkBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, String link, long stock)
                                                                                                             throws GoodsException {
        // 更新库存
        DubboResult<Long> stockRet = remoteStockBackendService.newStock(gid, stock);
        if (stockRet.isSuccess()) {
            Long batchId = goodsBatchService.createLinkBatch(gtype, gid, start, end, stockRet.getResult());
            if (GoodsTypeEnum.PLATFORM == gtype) {
                platformCouponGoodsService.editPlatformCouponType(gid, GoodsBatchEntity.BatchTypeLink);
            }
            
            goodsCouponService.importLinkCoupon(gtype, gid, batchId, getNewGoodsCouponId(gid), link);

            return batchId;
        } else {
            log.error("newStock error, because of:" + stockRet.getMsg());
            throw new GoodsException(ErrorCode.E0203003.getErrorCode(), stockRet.getMsg());
        }
    }

    @SuppressWarnings("deprecation")
    @Override
    public Long createRepeatBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, String code, String password,
                                  long stock) throws GoodsException {
        // 更新库存
        DubboResult<Long> stockRet = remoteStockBackendService.newStock(gid, stock);
        if (stockRet.isSuccess()) {
            Long batchId = goodsBatchService.createRepeatBatch(gtype, gid, start, end, stockRet.getResult());
            platformCouponGoodsService.editPlatformCouponType(gid, GoodsBatchEntity.BatchTypeRepeat);
            goodsCouponService.importRepeatCoupon(gtype, gid, batchId, getNewGoodsCouponId(gid), code, password);
            
            return batchId;
        } else {
            log.error("newStock error, because of:" + stockRet.getMsg());
            throw new GoodsException(ErrorCode.E0203003.getErrorCode(), stockRet.getMsg());
        }
    }

    @Override
    public Boolean updateLinkBatch(GoodsTypeEnum gtype, long gid, Long batchId, String link, long changeStock) {
        boolean success=goodsCouponService.updateLinkBatch(gtype, gid, batchId, link);
        if(!success){
            return success;
        }
        GoodsBatchEntity e=goodsBatchService.find(batchId);
        // 更新库存
        if(changeStock>0){
            DubboResult<Boolean> ret=remoteStockBackendService.increaseItemStock(e.getStockId(), changeStock);
            if(!ret.isSuccess()){
                return false;
            }
            //如果该批次已经标记为已使用，更新批次为未使用
            GoodsBatchEntity batch=goodsBatchService.find(batchId);
            if(batch.getStatus()==GoodsBatchEntity.StatusUsed){
                goodsBatchService.markBatchStatusNotUsed(batchId);
            }
        }else if(changeStock<0){
            DubboResult<Boolean> ret=remoteStockBackendService.decreaseItemStock(e.getStockId(), changeStock*(-1));
            if(!ret.isSuccess()){
                return false;
            }
        }
        return true;
    }

    @Override
    public Boolean updateRepeatBatch(GoodsTypeEnum gtype, long gid, Long batchId, String code, String password,
                                     long changeStock) {
        boolean success=goodsCouponService.updateRepeatBatch(gtype, gid, batchId, code, password);
        if(!success){
            return success;
        }
        GoodsBatchEntity e=goodsBatchService.find(batchId);
        // 更新库存
        if(changeStock>0){
            DubboResult<Boolean> ret=remoteStockBackendService.increaseItemStock(e.getStockId(), changeStock);
            if(!ret.isSuccess()){
                return false;
            }
            //如果该批次已经标记为已使用，更新批次为未使用
            GoodsBatchEntity batch=goodsBatchService.find(batchId);
            if(batch.getStatus()==GoodsBatchEntity.StatusUsed){
                goodsBatchService.markBatchStatusNotUsed(batchId);
            }
        }else if(changeStock<0){
            DubboResult<Boolean> ret=remoteStockBackendService.decreaseItemStock(e.getStockId(), changeStock*(-1));
            if(!ret.isSuccess()){
                return false;
            }
        }
        return success;
    }

    @Override
    public List<GoodsBatchDto> findBatchs(GoodsTypeEnum gtype, long gid) {
        List<GoodsBatchEntity> list=goodsBatchService.findNotDeletedBatchs(gtype, gid);
        List<GoodsBatchDto> ret=new ArrayList<>(list.size());
        for(GoodsBatchEntity e:list){
            GoodsBatchDto dto=convert(e);
            ret.add(dto);
            
            if(e.getBatchType()==GoodsBatchEntity.BatchTypeLink){
                GoodsCouponEntity c=goodsCouponService.findOneByGoodsBatchId(gtype, gid, e.getId());
                dto.setLink(c.getLink());
                
                if(e.getStockId()==null){
                    dto.setStock(0L);
                    dto.setTotalStock(0L);
                }else{
                    DubboResult<Long> stockRet=remoteStockBackendService.find(e.getStockId());
                    dto.setStock(stockRet.getResult());
                    DubboResult<Long> totalStockRet=remoteStockBackendService.findTotalStock(e.getStockId());
                    dto.setTotalStock(totalStockRet.getResult());
                }
            }else if(e.getBatchType()==GoodsBatchEntity.BatchTypeRepeat){
                GoodsCouponEntity c=goodsCouponService.findOneByGoodsBatchId(gtype, gid, e.getId());
                dto.setCode(c.getCode());
                dto.setPassword(c.getPassword());
                
                if(e.getStockId()==null){
                    dto.setStock(0L);
                    dto.setTotalStock(0L);
                }else{
                    DubboResult<Long> stockRet=remoteStockBackendService.find(e.getStockId());
                    dto.setStock(stockRet.getResult());
                    DubboResult<Long> totalStockRet=remoteStockBackendService.findTotalStock(e.getStockId());
                    dto.setTotalStock(totalStockRet.getResult());
                }
                
            }else if(e.getBatchType()==GoodsBatchEntity.BatchTypeNormal){
                if(e.getStockId()==null){
                    dto.setStock(0L);
                    dto.setTotalStock(0L);
                }else{
                    DubboResult<Long> stockRet=remoteStockBackendService.find(e.getStockId());
                    dto.setStock(stockRet.getResult());
                    DubboResult<Long> totalStockRet=remoteStockBackendService.findTotalStock(e.getStockId());
                    dto.setTotalStock(totalStockRet.getResult());
                }
            }
            
        }
        return ret;
    }

    private ExecutorService es=Executors.newCachedThreadPool();
    private AtomicInteger concurrent=new AtomicInteger();
    @Override
    public Long importNormalCoupons(final GoodsTypeEnum gtype,final long gid,final Long batchId, final String downloadUrl) {
        GoodsBatchImportLogEntity e=new GoodsBatchImportLogEntity();
        e.setGoodsBatchId(batchId);
        final Long importLogId=goodsBatchImportLogService.insert(e);
        
        int data=concurrent.incrementAndGet();
        if(data>5){
            concurrent.decrementAndGet();
            throw new RuntimeGoodsException(ErrorCode.E9999999);
        }
        
        es.submit(new Runnable() {
            
            @Override
            public void run() {
                try{
                    List<String> lines=downloadTxt(downloadUrl);
                    List<CouponFormat> cfs=new ArrayList<>(lines.size());
            
                    Map<String, String> couponMap=new HashMap<String, String>(lines.size());
                    
                    List<Long> goodsCouponIds=getNewBatchGoodsCouponIds(gid, lines.size());
                    String split="\t";
                    for(int i=0;i<lines.size();i++){
                        String l=lines.get(i);
                        String[] ss=l.split(split);
                        if (ss.length == 0) {
                            break;
                        }
                        String code=ss[0].trim();
                        String password = "";
                        if (ss.length == 2) {
                            password = ss[1];
                        }
                        
                        couponMap.put(code, password);
                        
                        CouponFormat cf=new CouponFormat();
                        cf.setCode(code);
                        cf.setGoodsCouponId(goodsCouponIds.get(i));
                        cf.setPassword(password);
                        
                        cfs.add(cf);
                    }
                    
                    GoodsBatchEntity batch=goodsBatchService.find(batchId);
                    Integer successCount=goodsCouponService.importNormalCoupons(batch, cfs);
                    Integer failCount=cfs.size()-successCount;
                    
                    
                    goodsBatchImportLogService.updateSuccessAndFail(importLogId, successCount, failCount);
                    if(successCount>0){
                        //如果该批次已经标记为已使用，更新批次为未使用
                        if(batch.getStatus()==GoodsBatchEntity.StatusUsed){
                            goodsBatchService.markBatchStatusNotUsed(batchId);
                        }
                    }
                    
                }catch(Exception e){
                    log.error("importNormalCoupons in thread error",e);
                }finally{
                    concurrent.decrementAndGet();
                }
            }
        });
        
        
        return importLogId;
    }

    @Override
    public GoodsBatchImportLogDto findBatchImportLog(Long goodsBatchImportLogId) {
        GoodsBatchImportLogEntity e=goodsBatchImportLogService.select(goodsBatchImportLogId);
        if(e!=null){
            GoodsBatchImportLogDto dto=new GoodsBatchImportLogDto();
            dto.setId(e.getId());
            dto.setFailCount(e.getFailCount());
            dto.setGoodsBatchId(e.getGoodsBatchId());
            dto.setSuccessCount(e.getSuccessCount());
            return dto;
        }
        
        return null;
    }

    @Override
    public Boolean deleteBatch(GoodsTypeEnum gtype, long gid, Long batchId) {
        boolean success=goodsBatchService.deleteBatch(gtype, gid, batchId);
        return success;
    }

    @Override
    public Page<GoodsCouponDto> findPage(GoodsTypeEnum gtype, long gid, long batchId, int pageSize, int pageIndex) {
        GoodsBatchEntity batch=goodsBatchService.find(batchId);
        DubboResult<Long> stockRet=remoteStockBackendService.findTotalStock(batch.getStockId());
        if(!stockRet.isSuccess()){
            Page<GoodsCouponDto> ret=new Page<>(pageSize, pageIndex);
            return ret;
        }
        
        Page<GoodsCouponEntity> page=goodsCouponService.findPage(gtype, gid, batchId, pageSize, pageIndex,stockRet.getResult().intValue());
        
        Page<GoodsCouponDto> ret=new Page<>(pageSize, pageIndex);
        ret.setTotalPages(page.getTotalPages());
        ret.setTotalCount(page.getTotalCount());
        
        List<GoodsCouponDto> list=new ArrayList<>(page.getList().size());
        for(GoodsCouponEntity e:page.getList()){
            list.add(convert(e));
        }
        ret.setList(list);
        
        return ret;
    }

    @Override
    public GoodsBatchDto findBatchStock(GoodsTypeEnum gtype, long gid, long batchId) {
        GoodsBatchEntity e=goodsBatchService.find(batchId);
        GoodsBatchDto dto=convert(e);
        if(e.getStockId()==null){
            dto.setStock(0L);
            dto.setTotalStock(0L);
            return dto;
        }
        DubboResult<Long> stockRet=remoteStockBackendService.find(e.getStockId());
        dto.setStock(stockRet.getResult());
        DubboResult<Long> totalStockRet=remoteStockBackendService.findTotalStock(e.getStockId());
        dto.setTotalStock(totalStockRet.getResult());
        if(totalStockRet.isSuccess() && stockRet.isSuccess()){
            dto.setStock(stockRet.getResult());
            dto.setTotalStock(totalStockRet.getResult());
        }else{
            throw new RuntimeGoodsException(ErrorCode.E9999999);
        }
            
        return dto;
    }
    
    @Override
    public List<GoodsCouponDto> findCouponByCode(GoodsTypeEnum gtype, long gid, long batchId, String code) {
        List<GoodsCouponEntity> list=goodsCouponService.searchByCode(gtype, gid, batchId, code);
        List<GoodsCouponDto> ret=new ArrayList<>();
        for(GoodsCouponEntity e:list){
            ret.add(convert(e));
        }
        return ret;
    }

    
    private Long getNewGoodsCouponId(Long gid){
        List<Long> ids=getNewBatchGoodsCouponIds(gid, 1);
        return ids.get(0);
    }
    
    private List<Long> getNewBatchGoodsCouponIds(Long gid,int count){
        DubboResult<Long> ret=remoteIDMakerBackendService.getBatchNextID(null, count);
        if(ret.isSuccess()){
            List<Long> ids=new ArrayList<>(count);
            for(int i=0;i<count;i++){
                ids.add(Long.valueOf((ret.getResult()+i)+IdSuffixTool.getTableSuffix(gid)));
            }
            return ids;
        }
        throw new RuntimeGoodsException(ErrorCode.E0203001);
    }
    

    private static BeanCopier GoodsBatchCopier=BeanCopier.create(GoodsBatchEntity.class, GoodsBatchDto.class, false);
    private GoodsBatchDto convert(GoodsBatchEntity e){
        GoodsBatchDto d=new GoodsBatchDto();
        GoodsBatchCopier.copy(e, d, null);
        return d;
    }
    private static BeanCopier GoodsCouponCopier=BeanCopier.create(GoodsCouponEntity.class, GoodsCouponDto.class, false);
    private GoodsCouponDto convert(GoodsCouponEntity e){
        GoodsCouponDto d=new GoodsCouponDto();
        GoodsCouponCopier.copy(e, d, null);
        return d;
    }
    
    private static List<String> downloadTxt(String url){
        File f=null;
        try {
            f=File.createTempFile("coupon-", "txt");
            URL httpurl = new URL(url);
           
            FileUtils.copyURLToFile(httpurl, f);  
            return Files.readLines(f, Charset.defaultCharset());
        } catch (Exception e) {  
            log.error("error",e);
        } finally{
            if(f!=null){
                if(!f.delete()){    
                    log.error("delete file fail");
                }
            }
        }
        
        return Collections.EMPTY_LIST;
    }

    @Override
    public Boolean updateValidDate(GoodsTypeEnum gtype, long acgId, long batchId, Date startDay, Date endDay) {
        return goodsBatchService.updateValidDate(gtype, acgId, batchId, startDay, endDay);
    }
}

