/**
 * Project Name:goods-center-biz File Name:GoodsBatchService.java Package Name:cn.com.duiba.goods.center.biz.service
 * Date:2016年5月25日下午12:21:02 Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.goods.center.biz.service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;

import java.util.Date;
import java.util.List;

/**
 * ClassName:GoodsBatchService <br/>
 * Date: 2016年5月25日 下午12:21:02 <br/>
 *
 * @author xuhengfei
 * @see
 * @since JDK 1.6
 */
public interface GoodsBatchService {
    /**
     * 创建一个普通批次
     *
     * @param gtype 商品类型
     * @param gid   商品id
     * @param start 开始时间
     * @param end   结束时间
     * @return 批次id
     * @author xuhengfei
     * @since JDK 1.6
     */
    Long createNormalBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, Long stockId);

    /**
     * 创建链接批次
     *
     * @param gtype   商品类型
     * @param gid     商品id
     * @param start   开始时间
     * @param end     结束时间
     * @param stockId 库存id
     * @return 批次id
     * @author xuhengfei
     * @since JDK 1.6
     */
    Long createLinkBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, Long stockId);

    /**
     * 创建重复券批次
     *
     * @param gtype   商品类型
     * @param gid     商品id
     * @param start   开始时间
     * @param end     结束时间
     * @param stockId 库存id
     * @return 批次id
     * @author xuhengfei
     * @since JDK 1.6
     */
    Long createRepeatBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, Long stockId);

    /**
     * 不包含已删除和已过期批次
     *
     * @param gtype 商品类型
     * @param gid   商品id
     * @return 结果集
     * @author xuhengfei
     * @since JDK 1.6
     */
    public List<GoodsBatchEntity> findNormalBatchs(GoodsTypeEnum gtype, long gid);

    /**
     * 不包含已删除的批次
     *
     * @param gtype 商品类型
     * @param gid   商品id
     * @return 结果集
     * @author xuhengfei
     * @since JDK 1.6
     */
    List<GoodsBatchEntity> findNotDeletedBatchs(GoodsTypeEnum gtype, long gid);

    /**
     * 逻辑删除批次
     *
     * @param gtype   商品类型
     * @param gid     商品id
     * @param batchId 批次id
     * @return 是否成功
     * @author xuhengfei
     * @since JDK 1.6
     */
    Boolean deleteBatch(GoodsTypeEnum gtype, long gid, Long batchId);

    /**
     * 因为唯一索引限制，物理删除批次数据<br/>
     * 此方法只被广告类型商品调用，其他地方不允许调用
     *
     * @param gtype   商品类型
     * @param gid     商品id
     * @param batchId 批次id
     * @return 是否成功
     * @author zufengze
     * @since JDK 1.6
     */
    Boolean physicalDeleteBatch(GoodsTypeEnum gtype, long gid, Long batchId);

    /**
     * 查询批次
     *
     * @param goodsBatchId 批次id
     * @return 批次信息
     * @author xuhengfei
     * @since JDK 1.6
     */
    GoodsBatchEntity find(Long goodsBatchId);

    /**
     * 获取商品的所有批次的剩余库存
     *
     * @param gtype        商品类型
     * @param gid          商品id
     * @param goodsBatchId 批次id
     * @return 库存数
     * @author xuhengfei
     * @since JDK 1.6
     */
    Long getSumBatchStock(GoodsTypeEnum gtype, long gid);

    /**
     * 获取商品的总库存(包含已经消耗的库存)
     *
     * @param gtype 商品类型
     * @param gid   商品id
     * @return 库存数
     * @author xuhengfei
     * @since JDK 1.6
     */
    Long getTotalAllBatchStock(GoodsTypeEnum gtype, long gid);

    /**
     * 刷新批次 当前批次如果已经售罄，标记当前批次已使用，切换到下一个批次
     *
     * @author xuhengfei
     * @param gtype
     * @param gid
     * @since JDK 1.6 public void refreshBatchUsing(GoodsTypeEnum gtype,long gid);
     */

    /**
     * 获取正在使用中的批次<br/>
     * 直接根据数据库中的状态来判断，不校验库存等信息
     *
     * @param gtype 商品类型
     * @param gid   商品id
     * @return 批次信息
     * @author xuhengfei
     * @since JDK 1.6
     */
    GoodsBatchEntity getUsingBatch(GoodsTypeEnum gtype, long gid);

    /**
     * 将批次标记为已经售罄<br/>
     * 由于批次只记录信息，不记录库存。当库存没有时，需要外部来设置售罄状态<br/>
     *
     * @param batchId 批次id
     * @author xuhengfei
     * @since JDK 1.6
     */
    void markBatchStatusUsed(long batchId);

    /**
     * 将批次标记为正在使用中<br/>
     * 由于批次只记录信息，不记录库存。当需要切换批次时，需要外部来设置售罄状态<br/>
     *
     * @param batchId 批次id
     * @author xuhengfei
     * @since JDK 1.6
     */
    void markBatchStatusUsing(long batchId);

    /**
     * 将批次标记为未使用<br/>
     * 由于批次只记录信息，不记录库存。当批次被再次导入券码时，需要外部设置为未使用，以便后续使用<br/>
     *
     * @param batchId 批次id
     * @author xuhengfei
     * @since JDK 1.6
     */
    void markBatchStatusNotUsed(long batchId);

    /**
     * 更新批次有效时间.
     *
     * @param gtype    商品类型
     * @param gid      商品ID
     * @param batchId  批次ID
     * @param startDay 生效期
     * @param endDay   失效期
     * @return true, if update valid date
     */
    Boolean updateValidDate(GoodsTypeEnum gtype, long gid, long batchId, Date startDay, Date endDay);

    /**
     * 更新批次类型,广告券使用 <br/>
     *
     * @param gtype   商品类型
     * @param gid     商品id
     * @param dstType
     * @return
     * @author sunjiangrong
     * @since JDK 1.6
     */
    boolean updateGoodsType(GoodsTypeEnum gtype, long gid, int dstType);

    /**
     * 查询多个商品批次
     *
     * @param gtype 商品类型
     * @param gids  商品id列表
     * @return 结果集
     */
    List<GoodsBatchEntity> findGoodsBatchs(GoodsTypeEnum gtype,
                                           List<Long> gids);

//    boolean updateStatusUsed(Long id);

    /**
     * @Author zjd
     * @Date 16/8/15
     */
    Long retrieveGoodsBatch(GoodsTypeEnum gtype, Long gid, Long batchId, Long dstQuantity) throws Exception;


    public boolean getBatchLock(Long goodsBatchId,Long gid,GoodsTypeEnum gtype);
    public void releaseBatchLock(Long goodsBatchId,Long gid,GoodsTypeEnum gtype);

//    public boolean updateStatus(Long id, int status);
}