/**
 * Project Name:goods-center-biz
 * File Name:GoodsCouponService.java
 * Package Name:cn.com.duiba.goods.center.biz.service
 * Date:2016年5月25日下午12:23:22
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.goods.center.biz.service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.tool.Page;
import cn.com.duiba.goods.center.biz.dao.GoodsCouponDao.CouponFormat;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;

import java.util.List;

/**
 * ClassName:GoodsCouponService <br/>
 * Date:     2016年5月25日 下午12:23:22 <br/>
 * @author   xuhengfei
 * @version
 * @since    JDK 1.6
 * @see
 */
public interface GoodsCouponService {
    /**
     * 更新链接券的批次上的链接信息
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid  商品id
     * @param batchId  批次id
     * @param link  链接内容
     * @return true/false
     * @since JDK 1.6
     */
    Boolean updateLinkBatch(GoodsTypeEnum gtype, long gid, Long batchId, String link);
    /**
     * 更新重复券的批次上的卡密信息
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid  商品id
     * @param batchId 批次id
     * @param code  卡号
     * @param password  密码
     * @return true/false 更新是否成功
     * @since JDK 1.6
     */
    Boolean updateRepeatBatch(GoodsTypeEnum gtype, long gid, Long batchId, String code, String password);

    /**
     * 导入普通券的券码
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid  商品id
     * @param batchId 批次id
     * @param coupons 券码列表
     * @return 成功导入条数
     * @since JDK 1.6
     */
    Integer importNormalCoupons(GoodsBatchEntity batch, List<CouponFormat> coupons);
    /**
     * 导入链接券的券码<br/>
     * 此方法只在创建链接批次时，自动执行插入一条记录
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid  商品id
     * @param goodsBatchId  批次id
     * @param goodsCouponId 券号
     * @param link 链接内容
     * @since JDK 1.6
     */
    void importLinkCoupon(GoodsTypeEnum gtype,long gid,long goodsBatchId,long goodsCouponId,String link);
    /**
     * 导入重复券的券码<br/>
     * 此方法只在创建重复批次是，自动执行插入一条记录
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param goodsBatchId 批次id
     * @param goodsCouponId 券号
     * @param code 卡号
     * @param password 密码
     * @since JDK 1.6
     */
    void importRepeatCoupon(GoodsTypeEnum gtype,long gid,long goodsBatchId,long goodsCouponId,String code,String password);
    /**
     * 分页查询券码信息
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @param pageSize 分页大小
     * @param pageIndex 页码(从1开始)
     * @param total 总共条数(请外部通过库存中心查询结果，然后传参进来)
     * @return 查询结果列表
     * @since JDK 1.6
     */
    Page<GoodsCouponEntity> findPage(GoodsTypeEnum gtype, long gid, long batchId, int pageSize,
                                                      int pageIndex,int total) ;


    List<GoodsCouponEntity> findPageByStatus(GoodsTypeEnum gtype, long gid, long batchId,int status,
                                     int start,int limit) ;

    /**
     * 查询一张优惠劵
     *
     * @author xuhengfei
     * @param goodsCouponId 券号
     * @return 券信息
     * @since JDK 1.6
     */
    GoodsCouponEntity find(Long goodsCouponId);
    /**
     * 查询剩余库存数
     * 改为私有方法，默认通过库存中心查询，不使用count查询
     *
     * @author xuhengfei
     * @param batch 批次对象
     * @return 剩余库存数
     * @since JDK 1.6
     */
//    long findNotUsedCount(GoodsBatchEntity batch);
    /**
     * 查询一个批次的所有库存数(包含已经消耗的)
     * 改为私有方法，默认通过库存中心查询，不使用count查询
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @return 库存数
     * @since JDK 1.6
     */
//    long findTotalAllCount(GoodsTypeEnum gtype,long gid,long batchId);
    /**
     * 从数据库获取一条优惠劵,并标记为已使用
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param batch 批次对象
     * @param consumerId 用户id
     * @param bizNum 业务编号
     * @return  券信息
     * @exception 售罄异常，其他异常
     * @since JDK 1.6
     */
    GoodsCouponEntity takeNormalCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch,long consumerId,String bizNum);
    /**
     * 领一张链接券
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param batch 批次对象
     * @param consumerId 用户id
     * @param bizNum 业务编号
     * @return 券信息
     * @since JDK 1.6
     */
    GoodsCouponEntity takeLinkCoupon(GoodsTypeEnum gtype,GoodsBatchEntity batch,long consumerId,String bizNum);
    /**
     * 领一张重复券
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param batch 批次对象
     * @param consumerId 用户id
     * @param bizNum 业务编号
     * @return 券信息
     * @since JDK 1.6
     */
    GoodsCouponEntity takeRepeatCoupon(GoodsTypeEnum gtype,GoodsBatchEntity batch,long consumerId,String bizNum);
    /**
     * 确认完成一张优惠劵的发放
     *
     * @author xuhengfei
     * @param couponId 券号
     * @param orderId 订单id
     * @since JDK 1.6
     */
    void completeCoupon(Long couponId, long orderId);
    /**
     * 回滚返还一张优惠劵
     *
     * @author xuhengfei
     * @param couponId 券号
     * @return 是否成功
     * @since JDK 1.6
     */
    Boolean rollbackNormalCoupon(Long couponId);


    /**
     *
     * 这个批次是否还有库存
     *
     * @author xuhengfei
     * @param gtype
     * @param gid
     * @param batchId
     * @return
     * @since JDK 1.6

    public boolean isBatchHaveStock(GoodsTypeEnum gtype,GoodsBatchEntity batch);
    */
    /**
     * 链接券和重复券，根据批次，查询信息
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param goodsBatchId 批次id
     * @return 券信息
     * @since JDK 1.6
     */
    GoodsCouponEntity findOneByGoodsBatchId(GoodsTypeEnum gtype,long gid,long goodsBatchId);
    /**
     * 根据批次id和卡号搜索
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid  商品id
     * @param goodsBatchId 批次id
     * @param code 卡号
     * @return 搜索结果集
     * @since JDK 1.6
     */
    List<GoodsCouponEntity> searchByCode(GoodsTypeEnum gtype,long gid,long goodsBatchId,String code);

    /**
     *
     * 删除某个优惠券指定批次所有券码. <br/>
     *
     * @author ZFZ
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @return 是否成功
     * @since JDK 1.6
     */
    boolean deleteGoodsCoupon(GoodsTypeEnum gtype, long gid, long batchId);

    int deleteGoodsCouponByIds(long gid ,List<Long> longs);
}

